package com.instabug.library.diagnostics

import com.instabug.library.Constants
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.diagnostics.customtraces.CustomTracesManager
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.customtraces.network.CustomTracesMappingKeys
import com.instabug.library.diagnostics.customtraces.utils.toTracesList
import com.instabug.library.diagnostics.di.DiagnosticsServiceLocator
import com.instabug.library.diagnostics.network.IBGDiagnosticsResponseCallback
import com.instabug.library.diagnostics.network.IBGDiagnosticsSyncManager
import com.instabug.library.diagnostics.nonfatals.NonFatalsManager
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator
import com.instabug.library.diagnostics.sdkEvents.onSuccessResolveSDKEvents
import com.instabug.library.networkv2.request.RequestParameter
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import org.json.JSONObject
import java.util.concurrent.Executor

class IBGDiagnosticsManagerImpl : IBGDiagnosticsManager {

    private val syncManager: IBGDiagnosticsSyncManager
        get() = DiagnosticsServiceLocator.getIBGDiagnosticsSyncManager()
    private val nonfatalsManager: NonFatalsManager? get() = ServiceLocator.getNonFatalsManager()
    private val settingsManager: SettingsManager
        get() = SettingsManager.getInstance()
    private val diagnosticsThreadExecutor: Executor
        get() = DiagnosticsServiceLocator.getDiagnosticsExecutor()
    private val customTracesManager: CustomTracesManager
        get() = CustomTracesServiceLocator.getCustomTracesManager()

    private val requestCallbacks = object :
        IBGDiagnosticsResponseCallback {
        override fun onSucceeded(requestParameters: List<RequestParameter<*>>) {
            InstabugSDKLogger.d(
                Constants.LOG_TAG,
                "Diagnostics synced successfully"
            )
            settingsManager.diagnosticsLastSyncTime = System.currentTimeMillis()
            nonfatalsManager?.clearCache()
            onSuccessResolveSDKEvents(requestParameters)
            customTracesManager.clearSyncedTraces((requestParameters.find { it.key == CustomTracesMappingKeys.CUSTOM_TRACES }?.value as? JSONObject)?.toTracesList())

        }

        override fun onFailed(throwable: Throwable) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Something went wrong while syncing Diagnostics" ,
                throwable
            )
        }
    }

    override fun sync() {

        if (InstabugCore.isFeatureEnabled(IBGFeature.INSTABUG)) {
            diagnosticsThreadExecutor.execute { syncManager.sync(requestCallbacks) }
        }
    }
}