package com.instabug.library.usersteps;

import static com.instabug.library.model.StepType.DISABLE;
import static com.instabug.library.model.StepType.DOUBLE_TAP;
import static com.instabug.library.model.StepType.ENABLE;
import static com.instabug.library.model.StepType.FLING;
import static com.instabug.library.model.StepType.LONG_PRESS;
import static com.instabug.library.model.StepType.MOVE;
import static com.instabug.library.model.StepType.PINCH;
import static com.instabug.library.model.StepType.TAP;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.view.ViewConfiguration;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.Platform;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.interactionstracking.IBGUINode;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.model.StepType;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.tracking.InstabugTrackingStepsProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.visualusersteps.TouchedView;

import java.lang.ref.WeakReference;
import java.util.concurrent.Future;

import kotlin.Pair;

/**
 * Created by tarek on 2/12/18.
 */

public class MotionEventRecognizer implements DefaultActivityLifeCycleEventHandler {

    @Nullable
    private GestureDetector gestureDetector;
    @Nullable
    private WeakReference<ScaleGestureDetector> scaleDetector;
    @Nullable
    private WeakReference<Activity> gestureDetectorReferencedActivityWeakReference;
    @Nullable
    private static MotionEventRecognizer instance;
    private final int LONG_PRESS_TIMEOUT;
    private final int SINGLE_TAP_TIMEOUT;
    private int CLICK_ACTION_THRESHOLD = 200;
    private float startX;
    private float startY;
    private long startTime = -1L;
    private long endTime = -1L;
    private boolean isLongPressLogged = false;
    private boolean doubleTapLogged = false;
    @Nullable
    private ActivityLifecycleSubscriber lifecycleSubscriber;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private MotionEventRecognizer() {
        if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q) {
            createGestureDetectorsWithAppContext();
        } else {
            // Try to create detectors with already-recorded Activity.
            // Just in case we missed the resume lifecycle event to do the creation
            // (in case of single activity model for example).
            createGestureDetectorsWithCurrentActivityContext();
            listenToActivityChanges();
        }

        LONG_PRESS_TIMEOUT = ViewConfiguration.getLongPressTimeout();
        SINGLE_TAP_TIMEOUT = /*ViewConfiguration.getTapTimeout();*/ 200;

    }

    private void createGestureDetectorsWithAppContext() {
        Context applicationContext = Instabug.getApplicationContext();
        if (applicationContext != null) {
            gestureDetector = new GestureDetector(applicationContext, new GestureListener());
            scaleDetector = new WeakReference<>(new ScaleGestureDetector(applicationContext, new PinchListener()));
        }
    }

    @RequiresApi(Build.VERSION_CODES.P)
    private void listenToActivityChanges() {
        if (lifecycleSubscriber != null) return;
        lifecycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
        lifecycleSubscriber.subscribe();
    }

    @Override
    public void handleActivityResumed() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            createGestureDetectorsWithCurrentActivityContext();
        }
    }

    @Override
    public void handleActivityDestroyed() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            cleanUpGestureDetectors();
        }

    }

    @RequiresApi(Build.VERSION_CODES.P)
    private void createGestureDetectorsWithCurrentActivityContext() {
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        Activity currentHeldActivityReference = null;
        if (gestureDetectorReferencedActivityWeakReference != null) {
            currentHeldActivityReference = gestureDetectorReferencedActivityWeakReference.get();
        }
        if (currentActivity != currentHeldActivityReference) {
            gestureDetector = null;
            scaleDetector = null;
            if (currentActivity != null) {
                gestureDetectorReferencedActivityWeakReference = new WeakReference<>(currentActivity);
                gestureDetector = new GestureDetector(currentActivity, new GestureListener());
                scaleDetector = new WeakReference<>(new ScaleGestureDetector(currentActivity, new PinchListener()));
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.P)
    private void cleanUpGestureDetectors() {
        if (gestureDetectorReferencedActivityWeakReference != null) {
            Activity activity = gestureDetectorReferencedActivityWeakReference.get();
            if (activity != null && activity.isDestroyed()) {
                gestureDetector = null;
                scaleDetector = null;
            }
        }
    }

    @VisibleForTesting
    static void release() {
        if (instance != null && instance.lifecycleSubscriber != null)
            instance.lifecycleSubscriber.unsubscribe();
        instance = null;
    }

    public static MotionEventRecognizer getInstance() {
        if (instance == null) {
            instance = new MotionEventRecognizer();
        }
        return instance;
    }

    public void recognize(MotionEvent event) {
        GestureDetector localGestureDetector = this.gestureDetector;
        if (this.scaleDetector != null) {
            ScaleGestureDetector localScaleDetector = this.scaleDetector.get();
            if (localGestureDetector != null) {
                localGestureDetector.onTouchEvent(event);
            }
            if (localScaleDetector != null) {
                localScaleDetector.onTouchEvent(event);
            }
        }
        tapDetector(event);
    }

    private void tapDetector(MotionEvent event) {
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                startX = event.getX();
                startY = event.getY();
                startTime = System.currentTimeMillis();
                isLongPressLogged = false;
                break;
            case MotionEvent.ACTION_UP:
                float endX = event.getX();
                float endY = event.getY();
                endTime = System.currentTimeMillis();
                if (isAClick(startX, endX, startY, endY)) {
                    if (isLongPress()) {
                        // long press confirmed
                        onViewTouched(LONG_PRESS, event);
                    } else if (!isLongPressLogged) {
                        if (!doubleTapLogged) {
                            // single tap confirmed
                            onViewTouched(TAP, event);
                        }
                    }
                    doubleTapLogged = false;
                }
                break;
            default:
                break;
        }
    }

    private boolean isAClick(float startX, float endX, float startY, float endY) {
        float differenceX = Math.abs(startX - endX);
        float differenceY = Math.abs(startY - endY);
        return !(differenceX > CLICK_ACTION_THRESHOLD/* =5 */ || differenceY > CLICK_ACTION_THRESHOLD);
    }

    /**
     * Indicates if the view was touched for a period more than {SINGLE_TAP_TIMEOUT}
     * This is a long press event but isn't detected by {@link GestureDetector.SimpleOnGestureListener#onLongPress(MotionEvent)}
     * as it isn't passed the LONG_PRESS_TIMEOUT yet
     *
     * @return true if the diff between {ACTION_DOWN} and {ACTION_UP} is more than SINGLE_TAP_TIMEOUT
     */
    private boolean isLongPress() {
        long differenceTime = endTime - startTime;
        return differenceTime > SINGLE_TAP_TIMEOUT && differenceTime < LONG_PRESS_TIMEOUT;
    }

    private void onViewTouched(@StepType String stepType, @Nullable MotionEvent ev) {
        if (ev != null) {
            onViewTouched(stepType, (int) ev.getRawX(), (int) ev.getRawY());
        }
    }

    @VisibleForTesting
    void onViewTouched(@StepType String stepType, float x, float y) {

        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            if (contract.fabBoundsContain((int) x, (int) y)) {
                return;
            }
        }

        Activity activity = InstabugInternalTrackingDelegate.getInstance().getTargetActivity();
        View decorView = null;
        if (activity != null) {
            decorView = activity.getWindow().getDecorView();
        }
        if (activity == null || decorView == null) return;

        Pair<IBGUINode, String> targetingResult;
        try {
            targetingResult =
                    CoreServiceLocator.getTargetUILocator().locate(decorView, x, y, stepType);
        } catch (Throwable e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while locating UI component", e);
            return;
        }
        if (targetingResult == null) return;

        final IBGUINode finalTarget = targetingResult.getFirst();
        final String finalStepType = targetingResult.getSecond();

        PoolProvider.postOrderedIOTask(Constants.USER_STEPS_EXECUTOR_KEY, () -> {
            if (finalTarget != null) {
                try {
                    addUserStep(finalTarget, finalStepType, activity);
                    addReproInteraction(finalTarget, finalStepType, activity);
                } catch (Throwable t) {
                    InstabugCore.reportError(t, "Error while processing steps");
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error while processing steps", t);
                }
            }
        });
    }

    private boolean isUserTrackingStepsEnable() {
        return InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.TRACK_USER_STEPS)
                == Feature.State.ENABLED;
    }

    @VisibleForTesting
    public void addUserStep(@NonNull IBGUINode target, @NonNull @StepType String gesture, @NonNull Context context) {
        if (!isUserTrackingStepsEnable()) return;
        int currentPlatform = SettingsManager.getInstance().getCurrentPlatform();
        boolean shouldDisableUserSteps = currentPlatform == Platform.FLUTTER && SettingsManager.shouldDisableNativeUserStepsCapturing();
        if (shouldDisableUserSteps && !isFlutterGestureAllowed(gesture,currentPlatform)) return;

        try {
            InstabugTrackingStepsProvider.getInstance()
                    .addUserStep(target.asUserStep(gesture, context));
        } catch (IllegalArgumentException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while adding touch user step");
        }
    }
    @SuppressWarnings("fb-contrib:LSC_LITERAL_STRING_COMPARISON")
    private boolean isFlutterGestureAllowed(@NonNull @StepType String gesture,int currentPlatform) {
        return currentPlatform == Platform.FLUTTER && (gesture.equals(StepType.APPLICATION_BACKGROUND) || gesture.equals(StepType.APPLICATION_FOREGROUND));
    }

    @WorkerThread
     private void addReproInteraction(@NonNull IBGUINode target, @NonNull @StepType String gesture, @NonNull Context context) {
        if (!CoreServiceLocator.getReproStepsProxy().isAuthorized()) return;
        try {
            Future<TouchedView> interactionLabelFuture = target.asTouchedView();
            if (interactionLabelFuture == null) return;
            String reportableGesture = gesture;
            if (target.isMovableWithProgress()) {
                reportableGesture = MOVE;
            }
            if (target.isCheckable()) {
                reportableGesture = target.isChecked() ? DISABLE : ENABLE;
            }
            CoreServiceLocator.getReproStepsProxy()
                    .addVisualUserStep(reportableGesture, context.getClass().getSimpleName(), target, interactionLabelFuture);
        } catch (IllegalArgumentException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while adding repro interaction");
        }
    }

    private class GestureListener extends GestureDetector.SimpleOnGestureListener {

        @Nullable
        private MotionEvent lastDownEvent;

        @Override
        public boolean onDown(MotionEvent e) {
            lastDownEvent = e;
            return super.onDown(e);
        }

        @Override
        public boolean onSingleTapConfirmed(MotionEvent e) {
            return false;
        }

        @Override
        public boolean onDoubleTap(MotionEvent e) {
            if (!doubleTapLogged) {
                // It's impossible to log double tab event without logging a single tap event before.
                // So, here we remove the redundant logged single tap event
                CoreServiceLocator.getReproStepsProxy().removeLastTapStep();
                onViewTouched(DOUBLE_TAP, e);
                doubleTapLogged = true;
            }
            return false;
        }

        @Override
        public void onLongPress(MotionEvent e) {
            if (!isLongPressLogged) {
                onViewTouched(LONG_PRESS, e);
                isLongPressLogged = true;
            }
        }

        @Override
        public boolean onFling(@Nullable MotionEvent e1, @NonNull MotionEvent e2, float velocityX, float velocityY) {
            MotionEvent e = e2 == null ? lastDownEvent : e2;
            onViewTouched(FLING, e);
            return false;
        }
    }

    private class PinchListener extends ScaleGestureDetector.SimpleOnScaleGestureListener {

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            onViewTouched(PINCH, detector.getFocusX(), detector
                    .getFocusY());
            return true;
        }
    }
}
