package com.instabug.library.tokenmapping

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.Instabug
import com.instabug.library.InstabugNetworkJob
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.request.*
import com.instabug.library.util.InstabugSDKLogger
import org.json.JSONObject

class TokenMappingSync(
    private val tokenMappingConfigs: TokenMappingConfigurations
) : InstabugNetworkJob() {

    private val mappedTokenReqCallback = object : Request.Callbacks<RequestResponse, Throwable> {
        override fun onSucceeded(response: RequestResponse?) {
            if (response == null || response.responseBody == null) return

            val responseBody = JSONObject(response.responseBody.toString())
            if (responseBody.isNull("token")) handleNewMappedToken(null)
            else handleNewMappedToken(responseBody.optString("token"))
        }

        override fun onFailed(error: Throwable?) {
            error ?: return
            val errorMsg = "Error while fetching mapped token"
            InstabugSDKLogger.e(Constants.LOG_TAG, errorMsg, error)
        }
    }

    private fun fetchMappedToken() {
        TokenMappingServiceLocator.getNetworkManager().doRequest(
            IBGNetworkWorker.CORE,
            RequestType.NORMAL,
            buildMappedTokenRequest(),
            mappedTokenReqCallback
        )
    }

    private fun buildMappedTokenRequest(): Request {
        return Request.Builder()
            .endpoint(Endpoints.MAPPED_TOKEN)
            .method(RequestMethod.GET)
            .tokenProvider(object : AppTokenProvider {
                override fun getAppToken(): String? {
                    return Instabug.getAppToken()
                }
            })
            .build()
    }

    @VisibleForTesting
    fun handleNewMappedToken(newMappedToken: String?) {
        if (!newMappedToken.isNullOrEmpty()
            && tokenMappingConfigs.mappedAppToken != newMappedToken
            && tokenMappingConfigs.isTokenMappingEnabled
        ) {
            tokenMappingConfigs.mappedAppToken = newMappedToken
            MappedTokenChangedHandler.onMappedTokenChanged()
        } else if (!tokenMappingConfigs.isTokenMappingEnabled) {
            tokenMappingConfigs.mappedAppToken = ""
        }
    }

    override fun start() {
        fetchMappedToken()
    }
}