package com.instabug.library.sessionreplay.monitoring

import com.instabug.library.internal.filestore.Cacheable
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.IBGLogsDisabled
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.IBGLogsUnavailable
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.NetworkLogsDisabled
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.NetworkLogsUnavailable
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.ReproScreenshotsDisabled
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.ReproScreenshotsUnavailable
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.ReproStepsDisabled
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.SREnabledInTheMiddleOfSession
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.UserStepsDisabled
import com.instabug.library.sessionreplay.monitoring.SRErrorCodes.UserStepsUnavailable
import com.instabug.library.util.extenstions.toMutableStringSet
import org.json.JSONArray
import org.json.JSONObject

data class SRAnalytics(
    val sessionId: String,
    var ibgLogsCount: Long = 0L,
    var networkLogsCount: Long = 0L,
    var userStepsCount: Long = 0L,
    var screenshotsMetadataCount: Long = 0L,
    var screenshotsCount: Long = 0L,
    var samplingDrops: Long = 0L,
    var sessionStorageViolationDrops: Long = 0L,
    var screenshotsStorageViolationDrops: Long = 0L,
    var aggregateStorageViolation: Boolean = false,
    val errors: MutableSet<String> = mutableSetOf(),
    var isSDKSampled: Boolean = false
) : Cacheable {
    override fun toJson(): JSONObject? = runCatching {
        JSONObject().apply {
            put(SessionID, sessionId)
            put(IBGLogsCount, ibgLogsCount)
            put(NetworkLogsCount, networkLogsCount)
            put(UserStepsCount, userStepsCount)
            put(ScreenshotsMetadataCount, screenshotsMetadataCount)
            put(ScreenshotsCount, screenshotsCount)
            put(SamplingDrops, samplingDrops)
            put(SessionStorageViolationDrops, sessionStorageViolationDrops)
            put(ScreenshotsStorageViolationDrops, screenshotsStorageViolationDrops)
            put(AggregateStorageViolation, aggregateStorageViolation)
            put(Errors, JSONArray(errors))
            put(SDKSampled, isSDKSampled)
        }
    }.getOrNull()

    companion object CachingKeys {
        private const val SessionID = "session_id"
        private const val IBGLogsCount = "ibg_logs_count"
        private const val NetworkLogsCount = "network_logs_count"
        private const val UserStepsCount = "user_steps_count"
        private const val ScreenshotsMetadataCount = "screenshots_metadata_count"
        private const val ScreenshotsCount = "screenshots_count"
        private const val SamplingDrops = "sampling_drops"
        private const val SessionStorageViolationDrops = "session_storage_violation_drops"
        private const val ScreenshotsStorageViolationDrops = "screenshots_storage_violation_drops"
        private const val AggregateStorageViolation = "aggregate_storage_violation"
        private const val Errors = "errors"
        private const val SDKSampled = "sdk_sampled"
    }

    object Helper {
        fun fromJson(json: JSONObject): SRAnalytics? = runCatching {
            with(json) {
                val analytics = takeIf { has(SessionID) }
                    ?.optString(SessionID)
                    ?.let { SRAnalytics(it) }
                takeIf { has(IBGLogsCount) }
                    ?.optLong(IBGLogsCount)
                    ?.let { analytics?.ibgLogsCount = it }
                takeIf { has(NetworkLogsCount) }
                    ?.optLong(NetworkLogsCount)
                    ?.let { analytics?.networkLogsCount = it }
                takeIf { has(UserStepsCount) }
                    ?.optLong(UserStepsCount)
                    ?.let { analytics?.userStepsCount = it }
                takeIf { has(ScreenshotsMetadataCount) }
                    ?.optLong(ScreenshotsMetadataCount)
                    ?.let { analytics?.screenshotsMetadataCount = it }
                takeIf { has(ScreenshotsCount) }
                    ?.optLong(ScreenshotsCount)
                    ?.let { analytics?.screenshotsCount = it }
                takeIf { has(SamplingDrops) }
                    ?.optLong(SamplingDrops)
                    ?.let { analytics?.samplingDrops = it }
                takeIf { has(SessionStorageViolationDrops) }
                    ?.optLong(SessionStorageViolationDrops)
                    ?.let { analytics?.sessionStorageViolationDrops = it }
                takeIf { has(ScreenshotsStorageViolationDrops) }
                    ?.optLong(ScreenshotsStorageViolationDrops)
                    ?.let { analytics?.screenshotsStorageViolationDrops = it }
                takeIf { has(AggregateStorageViolation) }
                    ?.optBoolean(AggregateStorageViolation)
                    ?.let { analytics?.aggregateStorageViolation = it }
                optJSONArray(Errors)
                    ?.toMutableStringSet()
                    ?.let { analytics?.errors?.addAll(it) }
                takeIf { has(SDKSampled) }
                    ?.optBoolean(SDKSampled)
                    ?.let { analytics?.isSDKSampled = it }
                analytics
            }
        }.getOrNull()
    }
}

class SRConfigurations(
    configurations: SRConfigurationsProvider,
    private val isSREnabledFromSessionStart: Boolean
) {
    private val isIBGLogsAvailable: Boolean = configurations.ibgLogsAvailable
    private val isIBGLogsEnabled: Boolean = configurations.ibgLogsRTEnabled
    private val isNetworkLogsAvailable: Boolean = configurations.networkLogsAvailable
    private val isNetworkLogsEnabled: Boolean = configurations.networkLogsRTEnabled
    private val isUserStepsAvailable: Boolean = configurations.userStepsAvailable
    private val isUserStepsEnabled: Boolean = configurations.userStepsRTEnabled
    private val isReproStepsEnabled = configurations.reproStepsRTEnabled
    private val isReproScreenshotsAvailable = configurations.isReproScreenshotsAvailable
    private val isReproScreenshotsEnabled = configurations.screenshotsRTEnabled

    fun generateErrorCodes() = buildSet {
        isSREnabledFromSessionStart.takeIf { !it }?.also { add(SREnabledInTheMiddleOfSession) }
        isIBGLogsAvailable.takeIf { !it }?.also { add(IBGLogsUnavailable) }
        isIBGLogsEnabled.takeIf { !it }?.also { add(IBGLogsDisabled) }
        isNetworkLogsAvailable.takeIf { !it }?.also { add(NetworkLogsUnavailable) }
        isNetworkLogsEnabled.takeIf { !it }?.also { add(NetworkLogsDisabled) }
        isUserStepsAvailable.takeIf { !it }?.also { add(UserStepsUnavailable) }
        isUserStepsEnabled.takeIf { !it }?.also { add(UserStepsDisabled) }
        isReproStepsEnabled.takeIf { !it }?.also { add(ReproStepsDisabled) }
        isReproScreenshotsAvailable.takeIf { !it }?.also { add(ReproScreenshotsUnavailable) }
        isReproScreenshotsEnabled.takeIf { !it }?.also { add(ReproScreenshotsDisabled) }
    }
}