package com.instabug.library.sessionreplay

import androidx.annotation.IntDef
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.AggregateStorageLimitViolation
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.EligibleForStoring
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.EnablementViolation
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.LimitsViolation
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.SamplingViolation
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.SessionStorageLimitViolation
import com.instabug.library.sessionreplay.SRLoggingController.Status.Companion.Unsaveable
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.model.SRLog
import com.instabug.library.sessionreplay.model.SRLogType
import com.instabug.library.sessionreplay.model.SRScreenshotLog
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.extenstions.logVerbose
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit
import kotlin.math.ceil

interface SRLoggingController {
    fun onSessionStarted(aggregateSize: Future<Long?>)
    fun onSessionEnded()
    fun onLogStored(bytesCount: Int)

    @Status
    fun getStatusForLog(log: SRLog): Int
    fun onScreenshotStored(bytesCount: Long)

    @Status
    fun getStatusForScreenshot(log: SRScreenshotLog): Int

    @IntDef(
        EligibleForStoring,
        EnablementViolation,
        LimitsViolation,
        Unsaveable,
        AggregateStorageLimitViolation,
        SessionStorageLimitViolation,
        SamplingViolation
    )
    /**
     * A class representing the possible statuses when a log or screenshot is being stored in SR.
     * Given all the limits of SR storage, statuses are grouped in three different classes:
     * 1. Success -> Represented by [EligibleForStoring], class binary (001-00000)
     * 2. Enablement violation -> Representing all possible negative statuses due to log type
     * enablement flags, whether there's a running session or not, etc .. class binary (010-00000)
     * 3. Limits violation -> Representing all possible negative statuses due to storage
     * or sampling violations, class binary (100-00000)
     * As binary classes suggest, each class of the three is represented with a left-most bit.
     * Each class has 5 right-most bits that can represent 5 different subclasses.
     * For example Limits violation class has:
     * 1. Aggregate storage violation subclass -> (100-00001)
     * 2. Session storage violation subclass -> (100-00010)
     * 3. Sampling violation subclass -> (100-00100)
     * This representation makes it easier that nested sealed classes.
     */
    annotation class Status {
        companion object {
            const val EligibleForStoring = 0x20
            const val EnablementViolation = 0x40
            const val LimitsViolation = 0x80
            const val Unsaveable = 0x100
            const val AggregateStorageLimitViolation = LimitsViolation or 0x1
            const val SessionStorageLimitViolation = LimitsViolation or 0x2
            const val SamplingViolation = LimitsViolation or 0x3
        }
    }
}

private const val SIZE_CONVERSION_FACTOR = 1024L

open class DefaultSRLoggingController(
    private val configurations: SRConfigurationsProvider
) : SRLoggingController {

    private var isSessionInProgress: Boolean = false
    private var samplingBaseTime: Long = -1L
    private var logsCountInCurrentSamplingPeriod = 0
    private var currentSessionLogsBytesCount = 0L
    private var currentSessionScreenshotsBytesCount = 0L
    private var aggregateStoredBytesCount = 0L

    override fun onSessionStarted(aggregateSize: Future<Long?>) {
        samplingBaseTime = TimeUtils.currentTimeMillis()
        logsCountInCurrentSamplingPeriod = 0
        currentSessionLogsBytesCount = 0L
        currentSessionScreenshotsBytesCount = 0L
        aggregateSize.get()
            ?.let { aggregateStoredBytesCount = it }
        "== Aggregate bytes count -> ${aggregateStoredBytesCount / (SIZE_CONVERSION_FACTOR * SIZE_CONVERSION_FACTOR)}MB(s)"
            .logVerbose(tag = SR_LOG_TAG)
        isSessionInProgress = true
    }

    override fun onSessionEnded() {
        isSessionInProgress = false
    }

    override fun onLogStored(bytesCount: Int) {
        logsCountInCurrentSamplingPeriod++
        currentSessionLogsBytesCount += bytesCount
    }

    @SRLoggingController.Status
    override fun getStatusForLog(log: SRLog): Int = when {
        !isSessionInProgress || !isOfAllowedType(log) -> EnablementViolation
        isMaxAggregateSizeExceeded() -> AggregateStorageLimitViolation
        isSessionMaxLogsSizeExceeded() -> SessionStorageLimitViolation
        isSamplingLimitsViolated() -> SamplingViolation
        else -> EligibleForStoring
    }

    override fun onScreenshotStored(bytesCount: Long) {
        currentSessionScreenshotsBytesCount += bytesCount
    }

    @SRLoggingController.Status
    override fun getStatusForScreenshot(log: SRScreenshotLog): Int =
        when {
            !isSessionInProgress || !isScreenshotsAllowed(log) -> EnablementViolation
            isMaxAggregateSizeExceeded() -> AggregateStorageLimitViolation
            isSessionMaxScreenshotsSizeExceeded() -> SessionStorageLimitViolation
            log.bitmap == null -> Unsaveable
            else -> EligibleForStoring
        }

    private fun isSamplingLimitsViolated(): Boolean {
        if (isSamplingDurationPassed()) resetSamplingData()
        return isSamplingDurationLimitExceeded()
    }

    private fun isSamplingDurationPassed(): Boolean {
        val currentTime = TimeUtils.currentTimeMillis()
        return (currentTime - samplingBaseTime) >= TimeUnit.SECONDS.toMillis(configurations.samplingRate.toLong())
    }

    private fun resetSamplingData() {
        samplingBaseTime = TimeUtils.currentTimeMillis()
        logsCountInCurrentSamplingPeriod = 0
    }

    private fun isSamplingDurationLimitExceeded(): Boolean {
        val isExceeded = logsCountInCurrentSamplingPeriod >= configurations.maxLogs
        if (isExceeded) "Logs/Screenshots storing is on cool down".logVerbose(tag = SR_LOG_TAG)
        return isExceeded
    }

    private fun isOfAllowedType(log: SRLog) = when (log.logType) {
        SRLogType.IBG_LOG -> configurations.ibgLogsEnabled
        SRLogType.NETWORK_LOG -> configurations.networkLogsEnabled
        SRLogType.USER_STEP -> configurations.userStepsEnabled
        SRLogType.SCREENSHOT -> configurations.isReproStepsEnabled
        else -> false
    }

    private fun isScreenshotsAllowed(log: SRLog): Boolean =
        log.logType == SRLogType.SCREENSHOT && configurations.isReproScreenshotsEnabled

    private fun isSessionMaxLogsSizeExceeded(): Boolean {
        val isExceeded = currentSessionLogsBytesCount >= toBytes(configurations.maxSessionSize)
        if (isExceeded) "Logs storing blocked (Max logs/session size reached)".logVerbose(tag = SR_LOG_TAG)
        return isExceeded
    }

    private fun isSessionMaxScreenshotsSizeExceeded(): Boolean {
        val isExceeded =
            currentSessionScreenshotsBytesCount >= toBytes(configurations.maxScreenshotsSizePerSession)
        if (isExceeded) "Screenshots storing blocked (Max screenshots/session size reached)"
            .logVerbose(tag = SR_LOG_TAG)
        return isExceeded
    }

    private fun isMaxAggregateSizeExceeded(): Boolean {
        val isExceeded = aggregateStoredBytesCount >= toBytes(configurations.maxSDKSize)
        if (isExceeded) "Logs/Screenshots storing blocked (Max aggregate reached)".logVerbose(tag = SR_LOG_TAG)
        return isExceeded
    }

    private fun toBytes(sizeInMB: Float): Long =
        ceil(sizeInMB * SIZE_CONVERSION_FACTOR * SIZE_CONVERSION_FACTOR).toLong()
}