package com.instabug.library.sessionprofiler;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.sessionprofiler.model.timeline.SessionProfilerTimeline;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import io.reactivexport.Observable;
import io.reactivexport.disposables.Disposable;

/**
 * Created by tarek on 3/4/18.
 */

public class SessionProfiler {

    private static SessionProfiler INSTANCE;

    private Boolean isFirstOOMSessionProfiler = false;

    private SessionProfilerTimeline timeline;
    @Nullable
    private Disposable capturingDisposable;

    private final AtomicBoolean isRunning = new AtomicBoolean(false);

    @Nullable
    private AttributesProvider attrsProvider;

    public synchronized static SessionProfiler getInstance() {
        if (INSTANCE != null) return INSTANCE;
        INSTANCE = new SessionProfiler();
        return INSTANCE;
    }

    private SessionProfiler() {
        timeline = new SessionProfilerTimeline();
        subscribeToSessionEvents();
    }

    private void subscribeToSessionEvents() {
        SessionStateEventBus.getInstance().subscribe(sessionState -> {
            if (sessionState == SessionState.START) {
                start();
            } else if (sessionState == SessionState.FINISH) {
                stop();
            }
        });
    }

    public synchronized void start() {
        if (!isFeatureEnabled() || isRunning.get()) return;
        attrsProvider = new AttributesProvider();
        capturingDisposable = Observable.interval(SessionProfilerTimeline.TIME_INTERVAL_HIGH_FREQUENCY, TimeUnit.MILLISECONDS)
                .map(iteration -> (iteration + 1) * SessionProfilerTimeline.TIME_INTERVAL_HIGH_FREQUENCY)
                .subscribe(this::capture, throwable -> InstabugSDKLogger.e(Constants.LOG_TAG, "Error while starting session profiler", throwable));
        isRunning.set(true);
    }

    public synchronized void stop() {
        if (!isRunning.get()) return;
        try {
            if (capturingDisposable != null) capturingDisposable.dispose();
            capturingDisposable = null;
        } catch (Throwable t) {
            // Swallow
        }
        attrsProvider = null;
        isRunning.set(false);
    }

    private boolean isFeatureEnabled() {
        return InstabugFeaturesManager.getInstance()
                .getFeatureState(IBGFeature.SESSION_PROFILER) == Feature.State.ENABLED;
    }

    private void capture(long time) {
        try {
            AttributesProvider localProvider = attrsProvider;
            if (localProvider == null) return;

            if (time % SessionProfilerTimeline.TIME_INTERVAL_LOW_FREQUENCY == 0) {

                //BatteryLevel
                try {
                    timeline.addBatteryState(localProvider.getBatteryState());
                } catch (AttributesProvider.NullContextException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't attach battery state (Null app context)");
                }

                //OrientationMode
                try {
                    timeline.addScreenOrientation(localProvider.getScreenOrientation());
                } catch (AttributesProvider.NullContextException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't attach orientation mode (Null app context)");
                }

                //ConnectivityState
                try {
                    timeline.addConnectivityState(localProvider.getConnectivityState());
                } catch (AttributesProvider.NullContextException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't attach network state (Null app context)");
                }
            }

            //MemoryUsage
            try {
                timeline.addMemoryUsage(localProvider.getMemoryUsage());
            } catch (AttributesProvider.NullContextException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't attach used memory (Null app context)");
            }

            //StorageUsage
            try {
                timeline.addStorageUsage(localProvider.getStorageUsage());
            } catch (AttributesProvider.NullContextException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't attach used storage (Null app context)");
            }

            timeline.trim();
        } catch (OutOfMemoryError e) {
            if (!isFirstOOMSessionProfiler) {
                IBGDiagnostics.reportNonFatal(e, "Couldn't capture session profiler. Device is low on memory ");
                isFirstOOMSessionProfiler = true;
            }
        }
    }

    public SessionProfilerTimeline fetch() {
        return fetch(1.0f);
    }

    public SessionProfilerTimeline fetch(float percentage) {
        return timeline.trim(percentage);
    }
}
