package com.instabug.library.session;

import androidx.annotation.IntRange;
import androidx.annotation.NonNull;

import com.instabug.library.model.session.CoreSession;
import com.instabug.library.model.session.SessionMapper;
import com.instabug.library.model.session.SessionRemoteEntity;
import com.instabug.library.model.session.SessionsBatchDTO;
import com.instabug.library.util.ListUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public final class SessionsBatcherImpl implements SessionBatcher {

    @NonNull
    @Override
    public List<SessionsBatchDTO> batch(@NonNull CoreSession session) {
        return batch(Collections.singletonList(session));
    }

    @NonNull
    @Override
    public List<SessionsBatchDTO> batch(@NonNull List<CoreSession> sessions) {
        int maxSessionsPerGroup = sessions.size();
        return batch(sessions, maxSessionsPerGroup);
    }

    @NonNull
    @Override
    public List<SessionsBatchDTO> batch(@NonNull List<CoreSession> sessions, @IntRange(from = 1) int maxPerBatch) {
        List<List<CoreSession>> sessionsGroups = ListUtils.split(sessions, maxPerBatch);
        return flatMap(sessionsGroups);
    }

    @NonNull
    private List<SessionsBatchDTO> flatMap(@NonNull List<List<CoreSession>> groupedBatches) {
        List<SessionsBatchDTO> batchesHolder = new ArrayList<>();
        for (List<CoreSession> batch : groupedBatches) {
            SessionsBatchDTO batchDTO = map(batch);
            batchesHolder.add(batchDTO);
        }
        return batchesHolder;
    }

    @NonNull
    private SessionsBatchDTO map(@NonNull List<CoreSession> batch) {
        SessionsBatchDTO batchDTO;
        switch (batch.size()) {
            case 0:
                Map<String, Object> emptyCommonKeys = Collections.emptyMap();
                List<SessionRemoteEntity> emptySessions = Collections.emptyList();
                batchDTO = SessionMapper.toDTO(emptyCommonKeys, emptySessions);
                break;
            case 1:
                CoreSession session = batch.get(0);
                SessionRemoteEntity remoteEntity = SessionMapper.toRemoteEntity(session);
                batchDTO = SessionMapper.toDTO(remoteEntity);
                break;
            default:
                batchDTO = SessionsMerger.merge(batch);
        }
        if (!batch.isEmpty()) {
            batchDTO.setProductionUsage(batch.get(batch.size() - 1).getProductionUsage());
        }
        return batchDTO;
    }
}
