package com.instabug.library.performanceclassification

import android.app.ActivityManager
import android.content.Context
import android.os.Build
import androidx.annotation.VisibleForTesting
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.Constants
import com.instabug.library.util.DeviceStateProvider
import com.instabug.library.util.InstabugSDKLogger
import java.io.RandomAccessFile
import java.util.Locale

class DevicePerformanceClassUtils(private val context: Context) {

    companion object {
        const val PERFORMANCE_CLASS_UNDEFINED = -1
        const val PERFORMANCE_CLASS_LOW = 0
        const val PERFORMANCE_CLASS_AVERAGE = 1
        const val PERFORMANCE_CLASS_HIGH = 2
    }

    private val activityManager: ActivityManager = getActivityManager()

    fun measureDevicePerformanceClass(): Int {
        var totalDeviceMemory: Long? = null
        var memoryClass: Int? = null
        val cpuCount = getCpuCount()
        var maxCpuFrequency: Int? = null

        try {
           val osVersion = BuildFieldsProvider.provideBuildVersion()
            totalDeviceMemory = getTotalMemory()

            if (
                osVersion < Build.VERSION_CODES.LOLLIPOP_MR1
                || cpuCount <= 2
                || totalDeviceMemory <= (2L * (1024L * 1024L * 1024L)) //Less than = 2GB memory.
            ) {
                logDeviceClassInfo("LOW", cpuCount, totalDeviceMemory)
                return PERFORMANCE_CLASS_LOW
            }

            memoryClass = activityManager.memoryClass
            maxCpuFrequency = getMaxCpuFrequency()
            val isCpuMeetAverageRequirements = cpuCount < 8 || maxCpuFrequency <= 2055

            if (
                osVersion < Build.VERSION_CODES.N
                || memoryClass <= 160
                || isCpuMeetAverageRequirements
            ) {
                logDeviceClassInfo(
                    "AVERAGE", cpuCount, totalDeviceMemory, memoryClass, maxCpuFrequency
                )
                return PERFORMANCE_CLASS_AVERAGE
            }
        } catch (e: OutOfMemoryError) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "OOM error while measuring device performance class", e
            )
        } catch (ex: Exception) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "Error while measuring device performance class", ex
            )
        }

        logDeviceClassInfo(
            "HIGH", cpuCount, totalDeviceMemory, memoryClass, maxCpuFrequency
        )
        return PERFORMANCE_CLASS_HIGH
    }

    fun isDevicePerformanceClassMatched(devices: Set<Int>): Boolean {
        val deviceHashCode = getDevicePerformanceClassHashCode()

        for (hashCode in devices)
            if (hashCode == deviceHashCode)
                return true

        return false
    }

    @VisibleForTesting
    fun getTotalMemory(): Long {
        val totalMemory = DeviceStateProvider.getTotalMemory(context)
        return if (totalMemory > -1) totalMemory * (1024 * 1024)
        else totalMemory
    }

    @VisibleForTesting
    fun getCpuCount(): Int {
        return Runtime.getRuntime().availableProcessors()
    }

    @VisibleForTesting
    fun getMaxCpuFrequency(): Int {
        val cpuMaxFreqInfoFilePath = "/sys/devices/system/cpu/cpu%d/cpufreq/cpuinfo_max_freq"
        val readMode = "r"
        var totalCpuFreq = 0
        var freqResolved = 0

        for (cpuNum in 0 until getCpuCount()) {
            try {
                val reader = RandomAccessFile(
                    String.format(Locale.ENGLISH, cpuMaxFreqInfoFilePath, cpuNum),
                    readMode
                )
                val line = reader.readLine()
                if (line != null) {
                    totalCpuFreq += line.toInt() / 1000
                    freqResolved++
                }
                reader.close()
            } catch (throwable: Throwable) {
                InstabugSDKLogger.v(
                    Constants.LOG_TAG, "Error while getting CPU frequency: $throwable"
                )
            }
        }

        return totalCpuFreq
    }
    @VisibleForTesting
    fun getDevicePerformanceClassHashCode(): Int {
     return if (BuildFieldsProvider.provideBuildDevice() != null && BuildFieldsProvider.provideBuildManufacturer() != null)
            (BuildFieldsProvider.provideBuildManufacturer() + BuildFieldsProvider.provideBuildDevice()).uppercase().hashCode()
        else
            -1
    }

    @VisibleForTesting
    fun getActivityManager(): ActivityManager {
        return context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
    }

    private fun logDeviceClassInfo(
        deviceClass: String,
        cpuCount: Int? = null,
        totalMemory: Long? = null,
        memoryClass: Int? = null,
        maxCpuFreq: Int? = null,
    ) {
        val totalMemoryInGb = totalMemory?.toDouble()?.div(1024 * 1024 * 1024)
        InstabugSDKLogger.v(
            Constants.LOG_TAG,
            "OS-Version: ${Build.VERSION.SDK_INT}, " +
                    "RAM: ${String.format("%.1f", totalMemoryInGb)}GB, " +
                    "CPU-Count: $cpuCount, " +
                    "MaxFreq: $maxCpuFreq, " +
                    "MemoryClass: $memoryClass, " +
                    "DeviceClass: $deviceClass"
        )
    }

}