package com.instabug.library.networkv2.service.userattributes;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.internal.utils.PreferencesUtils;
import com.instabug.library.model.UserAttribute;
import com.instabug.library.model.UserAttributes;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.TaskDebouncer;
import com.instabug.library.util.TimeUtils;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;


class AttributesRemoteDataSource {

    @VisibleForTesting
    static final String KEY_LAST_SYNC = "key_user_attrs_last_sync";
    @VisibleForTesting
    static final String KEY_TTL = "key_user_attrs_ttl";
    @VisibleForTesting
    static final String KEY_HASH = "key_user_attrs_hash";
    private final PreferencesUtils preferencesUtils;
    private final NetworkManager networkManager;
    private final TaskDebouncer networkDebouncer = new TaskDebouncer(TimeUnit.SECONDS.toMillis(2));

    AttributesRemoteDataSource(NetworkManager networkManager, PreferencesUtils preferencesUtils) {
        this.preferencesUtils = preferencesUtils;
        this.networkManager = networkManager;
    }

    /**
     * Queries all user attributes form remote
     *
     * @param request the request object generated by {@link #getRequest(String, String, String)}
     *                request status code was {@link HttpStatusCode._2xx#OK}
     */
    public void query(final Request request, final Request.Callbacks<List<UserAttribute>, Throwable> callbacks) {
        networkDebouncer.debounce(new Runnable() {
            @Override
            public void run() {
                doRequest(request, callbacks);
            }
        });
    }

    /**
     * Generates a request object containing all needed info (i.e. endpoint, method, headers, etc..)
     * and appends the passed parameters to url parameters
     *
     * @param email the currently identified email
     * @return a prepared request object ready for network use
     * @see AttributesDataMapper#toRequest(String, String)
     */
    public Request getRequest(String email) {
        return AttributesDataMapper.toRequest(email, getHash());
    }

    /**
     * Checks whether or not the TTL has passed and returns a network observable if passed or empty
     * Observable otherwise
     *
     * @param request the request object to fetch user attributes from remote
     *                the elapsed time since last fetch has passed the TTL
     * @see #getTTL()
     * @see #getLastSyncTimestamp()
     * @see #isEligibleToDoRequest(long)
     */
    void doRequest(Request request, final Request.Callbacks<List<UserAttribute>, Throwable> callbacks) {
        if (isEligibleToDoRequest(TimeUtils.currentTimeMillis()) && request != null) {
            networkManager.doRequest(IBGNetworkWorker.CORE, RequestType.NORMAL, request, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse response) {
                    if (response != null && response.getResponseCode() < RequestResponse.HttpStatusCode._4xx.BAD_REQUEST) {
                        updateLastSyncTimestamp(TimeUtils.currentTimeMillis());

                        if (response.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                            String hash = response.getHeaders().get(Header.IF_MATCH);
                            updateHash(hash);

                            String jsonString = AttributesDataMapper.toJson(response);
                            UserAttributes attributes;
                            try {
                                attributes = AttributesDataMapper.toUserAttributes(jsonString);
                            } catch (JSONException e) {
                                attributes = null;
                                callbacks.onFailed(e);
                            }

                            if (attributes != null) {
                                long ttl = TimeUnit.SECONDS.toMillis(attributes.getTtl());
                                updateTTL(ttl);

                                Map<String, String> attributesMap = attributes.getMap();
                                if (attributesMap != null) {
                                    List<UserAttribute> attributesList = AttributesDataMapper.toUserAttributeList(attributesMap);
                                    callbacks.onSucceeded(attributesList);
                                } else {
                                    callbacks.onSucceeded(new ArrayList<UserAttribute>());
                                }
                            }
                        }
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    callbacks.onFailed(error);
                }
            });
        }
    }

    @VisibleForTesting
    void updateLastSyncTimestamp(long timestamp) {
        preferencesUtils.saveOrUpdateLong(KEY_LAST_SYNC, timestamp);
    }

    @VisibleForTesting
    long getLastSyncTimestamp() {
        return preferencesUtils.getLong(KEY_LAST_SYNC);
    }

    @VisibleForTesting
    void updateTTL(long ttl) {
        preferencesUtils.saveOrUpdateLong(KEY_TTL, ttl);
    }

    @VisibleForTesting
    long getTTL() {
        return preferencesUtils.getLong(KEY_TTL);
    }

    @VisibleForTesting
    void updateHash(@Nullable String hash) {
        preferencesUtils.saveOrUpdateString(KEY_HASH, hash);
    }

    @VisibleForTesting
    @Nullable
    String getHash() {
        return preferencesUtils.getString(KEY_HASH);
    }

    /**
     * Checks whether or not the TTL has been satisfied since last sync until the
     * {@code requestTimestamp}
     *
     * @param requestTimestamp the timestamp of the do request action
     * @return true if the elapsed time was greater than the ttl or false otherwise
     * @see #getTTL()
     * @see #getLastSyncTimestamp()
     * @see #doRequest(Request, Request.Callbacks)
     */
    @VisibleForTesting
    boolean isEligibleToDoRequest(long requestTimestamp) {
        long elapsedTime = requestTimestamp - getLastSyncTimestamp();
        return elapsedTime > getTTL();
    }
}
