package com.instabug.library.networkv2;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import androidx.annotation.VisibleForTesting;

import com.instabug.library.BuildConfig;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.networkv2.request.Constants;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.util.InstabugSDKLogger;


/**
 A utils class contains network layer helper methods.
 */

public class NetworkUtils {
    /**
     Check device connectivity before performing any request.

     @return {@code true} if network connectivity exists or is in the process of being established,
     {@code false} otherwise.
     */
    public static boolean isOnline(Context context) {
        try {
            ConnectivityManager connectivityManager;

            if (context != null) {
                connectivityManager = (ConnectivityManager) context
                        .getSystemService(Context.CONNECTIVITY_SERVICE);

                NetworkInfo activeNetworkInfo;
                if (connectivityManager != null) {
                    activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
                    if (activeNetworkInfo != null && activeNetworkInfo.isConnectedOrConnecting()) {
                        return true;
                    }
                }
            }
        } catch (SecurityException e) {
            InstabugSDKLogger.w("NetworkManager", "Could not read network state. To enable " +
                    "please add the following line in your AndroidManifest.xml <uses-permission " +
                    "android:name=\"android.permission.ACCESS_NETWORK_STATE\"/>\n" + e.getMessage
                    ());
        } catch (Exception e) {
            InstabugSDKLogger.e("NetworkManager", "Something went wrong while checking " +
                    "network state", e);
        }
        return false;
    }

    /**
     Check if it the request is the SDK features (App settings)request.

     @return {@code true} if request endpoint was defined and contains {@code Endpoint.CORE.APP_SETTINGS},
     {@code false} otherwise.
     */
    @VisibleForTesting
    public static boolean isFeatureRequest(String url) {
        return url.contains(Endpoints.APP_SETTINGS);
    }

    /**
     Check if it the request authorization feature is available.

     @return {@code true} if the authorization feature is available and enabled,
     {@code false} otherwise.

     @see InstabugFeaturesManager#isFeatureAvailable
     @see InstabugFeaturesManager#getFeatureState
     */

    public static boolean isAuthorizationEnabled() {
        return InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.BE_DISABLE_SIGNING) &&
                InstabugFeaturesManager.getInstance().getFeatureAbsoluteState(IBGFeature.BE_DISABLE_SIGNING) != Feature.State.DISABLED;
    }

    /**
     Check if it the request authorization is required.

     @return {@code true} if request is an Instabug APIs request and is not SDK features (App settings)request,
     {@code false} otherwise.

     @see #isInstabugRequest
     @see #isFeatureRequest
     */
    public static boolean isAuthorizationRequired(String url) {
        return isInstabugRequest(url) && !isFeatureRequest(url);
    }

    /**
     Check if it the request is Instabug APIs request.

     @return {@code true} if a request is hitting Instabug SDK APIs,
     {@code false} otherwise.

     @see Constants#BASE_URL , {@link Constants#APM_BASE_URL}
     @param url
     */
    public static boolean isInstabugRequest(String url) {
        return url.contains(Constants.BASE_URL) || url.contains(Constants.APM_BASE_URL) || url.contains(BuildConfig.DIAGNOSIS_DOMAIN);
    }
}
