package com.instabug.library.model;


import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;
import com.instabug.library.util.memory.MemoryUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

/**
 * The type Console log.
 *
 * @author Hossam
 */
public class ConsoleLog implements Cacheable, Serializable {
    private static final String CONSOLE_LOG_DEFAULT_DATE_FORMAT = "MM-dd HH:mm:ss.SSS";
    private static final String KEY_TIMESTAMP = "timestamp";
    private static final String KEY_MESSAGE = "message";
    private static final String KEY_DATE = "date";
    public static final int LINES_LIMIT = 700;

    @Nullable
    private String message;
    private long timeStamp;
    @Nullable
    private String date;

    /**
     * From json array list.
     *
     * @param consoleLogsAsJsonArray the console logs as json array
     * @return the ConsoleLogs array list
     * @throws JSONException the json exception
     */
    public static ArrayList<ConsoleLog> fromJson(JSONArray consoleLogsAsJsonArray) throws JSONException {
        ArrayList<ConsoleLog> consoleLogs = new ArrayList<>();
        if (consoleLogsAsJsonArray != null && consoleLogsAsJsonArray.length() > 0) {
            for (int i = 0; i < consoleLogsAsJsonArray.length(); i++) {
                ConsoleLog consoleLog = new ConsoleLog();
                consoleLog.fromJson(consoleLogsAsJsonArray.getJSONObject(i).toString());
                consoleLogs.add(consoleLog);
            }
        }
        return consoleLogs;
    }

    public static JSONArray toJson(@Nullable ArrayList<ConsoleLog> consoleLogsArrayList) {
        JSONArray jsonArray = new JSONArray();
        if (consoleLogsArrayList != null && consoleLogsArrayList.size() > 0) {
            for (ConsoleLog consoleLog : consoleLogsArrayList) {
                try {
                    jsonArray.put(new JSONObject(consoleLog.toJson()));
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error while parsing console log " + e.getMessage());
                }
            }
        }
        return jsonArray;
    }

    /**
     * Gets time stamp.
     *
     * @return the time stamp
     */
    public long getTimeStamp() {
        return timeStamp;
    }

    /**
     * Sets time stamp.
     *
     * @param timeStamp the time stamp
     */
    public void setTimeStamp(long timeStamp) {
        this.timeStamp = timeStamp;
    }

    /**
     * Gets message.
     *
     * @return the message
     */
    @Nullable
    public String getMessage() {
        return message;
    }

    /**
     * Sets message.
     *
     * @param message the message
     */
    public void setMessage(@Nullable String message) {
        this.message = message;
    }

    @Nullable
    public String getDate() {
        return date;
    }

    public void setDate(String date) {
        this.date = date;
    }

    /**
     * toJson converts data model into Json object
     *
     * @return ConsoleLog as JSONObject
     * @throws JSONException the json exception
     */
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {

        if (timeStamp == 0 && Looper.myLooper() != Looper.getMainLooper()) {
            setTimeStamp(convertConsoleLogMessageDateToTimeStamp(getDate(),
                    new SimpleDateFormat(
                            ConsoleLog.CONSOLE_LOG_DEFAULT_DATE_FORMAT, Locale.US),
                    Calendar.getInstance(),
                    Calendar.getInstance()));
        }

        JSONObject actionJsonObject = new JSONObject();
        actionJsonObject.put(KEY_TIMESTAMP, getTimeStamp());
        actionJsonObject.put(KEY_MESSAGE, getMessage());
        actionJsonObject.put(KEY_DATE, getDate());
        return actionJsonObject.toString();
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject consoleLogAsJsonObject = new JSONObject(modelAsJson);
        if (consoleLogAsJsonObject.has(KEY_TIMESTAMP))
            if (StringUtility.isNumeric(consoleLogAsJsonObject.getString(KEY_TIMESTAMP))) {
                setTimeStamp(consoleLogAsJsonObject.getLong(KEY_TIMESTAMP));
            } else {
                setTimeStamp(convertConsoleLogMessageDateToTimeStamp(
                        consoleLogAsJsonObject.getString(KEY_TIMESTAMP), new SimpleDateFormat(
                                ConsoleLog.CONSOLE_LOG_DEFAULT_DATE_FORMAT, Locale.US),
                        Calendar.getInstance(),
                        Calendar.getInstance()));
            }
        if (consoleLogAsJsonObject.has(KEY_MESSAGE))
            setMessage(consoleLogAsJsonObject.getString(KEY_MESSAGE));

        if (consoleLogAsJsonObject.has(KEY_DATE))
            setDate(consoleLogAsJsonObject.getString(KEY_DATE));
    }

    @NonNull
    @Override
    public String toString() {
        return "ConsoleLog{" +
                "timeStamp='" + timeStamp + '\'' +
                ", message='" + message + '\'' +
                ", date='" + date + '\'' +
                '}';
    }

    static long convertConsoleLogMessageDateToTimeStamp(@Nullable String dateAsString, SimpleDateFormat dateFormat, Calendar tempCalendar, Calendar consoleLogMessageCalendar) {
        if (dateAsString != null) {
            try {
                Date date = dateFormat.parse(dateAsString);
                if (date != null) {
                    tempCalendar.setTime(date);
                }
                consoleLogMessageCalendar.set(Calendar.MONTH, tempCalendar.get(Calendar.MONTH));
                consoleLogMessageCalendar.set(Calendar.DAY_OF_MONTH, tempCalendar.get(Calendar.DAY_OF_MONTH));
                consoleLogMessageCalendar.set(Calendar.HOUR_OF_DAY, tempCalendar.get(Calendar.HOUR_OF_DAY));
                consoleLogMessageCalendar.set(Calendar.MINUTE, tempCalendar.get(Calendar.MINUTE));
                consoleLogMessageCalendar.set(Calendar.SECOND, tempCalendar.get(Calendar.SECOND));
                consoleLogMessageCalendar.set(Calendar.MILLISECOND, tempCalendar.get(Calendar.MILLISECOND));
                return consoleLogMessageCalendar.getTimeInMillis();
            } catch (Exception e) {
                // silent exception we should skip un-parselable messages that may happen like proces start
                // logs or unicode body prints
                return 0;
            }
        }

        return 0;
    }

    @Nullable
    public synchronized static List<String> getConsoleLogs(float percentage) {

        Context context = Instabug.getApplicationContext();
        if (context != null && MemoryUtils.isLowMemory(context)) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Running low on memory. Excluding Console Logs serialization from state builder.");
            return null;
        }
        LinkedList<String> consoleLogsArray = new LinkedList<>();

        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.CONSOLE_LOGS)
                == Feature.State.ENABLED) {
            int maxNumLines = CoreServiceLocator.getLimitConstraintApplier().applyConstraints(Math.round(LINES_LIMIT * percentage));
            java.lang.Process process = null;
            try {
                process = Runtime.getRuntime().exec("logcat -v time -d -t " + maxNumLines +
                        " --pid=" + android.os.Process.myPid());
            } catch (Throwable e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Could not read logcat log", e);
                return consoleLogsArray;
            }
            BufferedReader bufferedReader = new BufferedReader(new
                    InputStreamReader(process.getInputStream(), Charset.forName("UTF-8")));
            try {
                String line;
                while ((line = bufferedReader.readLine()) != null) {
                    if (line.length() < 19) continue;
                    if (consoleLogsArray.size() >= maxNumLines) consoleLogsArray.removeFirst();
                    consoleLogsArray.add(line);
                }
                return consoleLogsArray;
            } catch (Throwable e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Could not read logcat log", e);
                return consoleLogsArray;
            } finally {
                process.destroy();
                try {
                    bufferedReader.close();
                } catch (IOException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to close file reader", e);
                }
            }
        } else {
            return consoleLogsArray;
        }
    }
}
