package com.instabug.library.logging;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.UserEventsEventBus;
import com.instabug.library.internal.storage.cache.user.UserCacheManager;
import com.instabug.library.internal.storage.cache.user.UserDbHelper;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.user.UserEvent;
import com.instabug.library.user.UserEventParam;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

public class InstabugUserEventLogger {

    private static volatile InstabugUserEventLogger instabugUserEventLogger;
    private static final int USER_EVENT_COUNT_LIMIT = 1_000;
    private List<UserEvent> userEvents;
    private ConcurrentHashMap<String, Integer> userEventsCount;

    private InstabugUserEventLogger() {
        userEvents = new CopyOnWriteArrayList<>();
        userEventsCount = new ConcurrentHashMap<>();
    }

    public static synchronized InstabugUserEventLogger getInstance() {
        if (instabugUserEventLogger == null)
            instabugUserEventLogger = new InstabugUserEventLogger();

        return instabugUserEventLogger;
    }

    /**
     * log @{@link UserEvent} associated with the current user
     *
     * @param eventIdentifier identifier to identify this UserEvent
     * @param userEventParams params  associated with this event
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 4.0
     */
    public synchronized void logUserEvent(@NonNull final String eventIdentifier, final UserEventParam... userEventParams) {
        PoolProvider.getUserActionsExecutor().execute(
                new Runnable() {
                    @Override
                    public void run() {
                        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.USER_EVENTS) == Feature
                                .State.ENABLED) {
                            UserEvent userEvent = new UserEvent()
                                    .setEventIdentifier(eventIdentifier)
                                    .setDate(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds());
                            for (UserEventParam userEventParam : userEventParams) {
                                userEvent.addParam(userEventParam);
                            }
                            if (userEvents.size() >= USER_EVENT_COUNT_LIMIT) {
                                userEvents.remove(0);
                            }
                            userEvents.add(userEvent);

                            Integer lastUserEventsCount = userEventsCount.get(eventIdentifier);

                            if (lastUserEventsCount != null) {
                                userEventsCount.put(eventIdentifier, ++lastUserEventsCount);
                            } else {
                                userEventsCount.put(eventIdentifier, 1);
                            }

                                runInsertionHandlerAvailable(UserManager.getUUIDBlocking(), !UserManager.isLoggedIn());


                        }
                    }
                }
        );
    }
    @WorkerThread
    private void runInsertionHandlerAvailable(final String uuid , final boolean anonymous) {
        Context context = Instabug.getApplicationContext();
        if (context != null ) {
          try {

                  for (ConcurrentHashMap.Entry<String, Integer> entry : userEventsCount.entrySet()) {
                      incrementEventLoggingCount(entry.getKey(), entry.getValue(), uuid, anonymous);
                  }
                  userEventsCount.clear();
              }
             catch (Throwable e) {
              InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + "while inserting user events");

          }
      }

    }

    /**
     * get logging count for all @{@link UserEvent} associated with #userEventIdentifier and
     * current user
     *
     * @param userEventIdentifier identifier to query a specific @{@link UserEvent}
     * @return list  of @{@link UserEvent}
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 4.0
     */
    @VisibleForTesting
    int getLoggingEventCount(@NonNull String userEventIdentifier) {
        return UserEventsDbHelper.getOccurrencesCount(userEventIdentifier, UserManager.getUUID());
    }

    /**
     * get all @{@link UserEvent} associated with current user}
     *
     * @return list  of @{@link UserEvent}
     * @throws IllegalStateException if Instabug object wasn't built using
     *                               {@link Instabug.Builder#build()} before this method was called
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 4.0
     */
    public List<UserEvent> getUserEvents() throws IllegalStateException {
        return userEvents;
    }

    public List<UserEvent> getUserEvents(float percentage) throws IllegalStateException {
        int numberOfDesiredEvents = Math.round(USER_EVENT_COUNT_LIMIT * percentage);
        if (userEvents.size() <= numberOfDesiredEvents) return userEvents;
        int startIndex = userEvents.size() - numberOfDesiredEvents;
        return userEvents.subList(startIndex, userEvents.size());
    }

    /**
     * clear all @{@link UserEvent} associated with current user from memory
     *
     * @throws IllegalStateException if Instabug object wasn't built using
     *                               {@link Instabug.Builder#build()} before this method was called
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 4.0
     */
    public void clearAll() throws IllegalStateException {
        userEvents.clear();
    }

    /**
     * clear all @{@link UserEvent} associated with current user from database
     *
     * @see UserEvent
     * @see Instabug.Builder#build()
     * @since 4.0.4
     */
    public void clearLoggingData() throws IllegalStateException {
        UserEventsDbHelper.deleteAll(UserManager.getUUID());
    }

    private void incrementEventLoggingCount(@NonNull String eventIdentifier, int incrementBy,
                                            String uuid, boolean anonymous) {
        int eventLoggingCount = UserEventsDbHelper.getOccurrencesCount(eventIdentifier, UserManager.getUUID());
        eventLoggingCount += incrementBy;
        if(UserDbHelper.retrieve(uuid) == null) {
            UserCacheManager.insertIfNotExists(uuid, SettingsManager.getInstance().getSessionsCount());
        }
        UserEventsDbHelper.insertOrUpdate(eventIdentifier, eventLoggingCount, uuid, anonymous);
        UserEvent userEvent = new UserEvent()
                .setEventIdentifier(eventIdentifier);
        UserEventsEventBus.getInstance().post(userEvent);
    }
}
