package com.instabug.library.internal.storage.cache.db.userAttribute;

import static com.instabug.library.util.filters.AttributeFiltersFunctions.isValidAttribute;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.orchestrator.InsertUserAttributeAction;
import com.instabug.library.internal.orchestrator.InsertUuidAction;
import com.instabug.library.model.UserAttribute;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.threading.PoolProvider;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class UserAttributeCacheManager {

    public static void insert(String key, String value) {
        if (!isValidAttribute(key, value)) return;
        String uuid = UserManager.getUUID();
        UserAttribute userAttribute = new UserAttribute.Builder(key, value)
                .type(UserAttribute.Type.SDK)
                .anonymous(!UserManager.isLoggedIn())
                .uuid(uuid)
                .build();

        ActionsOrchestrator.obtainOrchestrator(PoolProvider.getUserActionsExecutor())
                .addSameThreadAction(new InsertUuidAction(
                        uuid,
                        UserManager.getSessionsCount()))
                .addSameThreadAction(new InsertUserAttributeAction(userAttribute))
                .orchestrate();
    }

    @Nullable
    public static String retrieve(String key) {
        return UserAttributesDbHelper.retrieve(key, UserManager.getUUID());
    }

    @NonNull
    public static HashMap<String, String> retrieveAll() {
        return UserAttributesDbHelper.retrieveAll(UserManager.getUUID());
    }

    public static HashMap<String, String> retrieveAllSDKAttributes() {
        return UserAttributesDbHelper.retrieveAllSDKAttributes(UserManager.getUUID());
    }

    @NonNull
    public static List<UserAttribute> retrieveAnonymousUserAttributes() {
        return UserAttributesDbHelper.retrieveAnonymousUserAttribute();
    }

    @UserAttribute.Type
    public static int getType(String key) {
        return UserAttributesDbHelper.getType(key, UserManager.getUUID());
    }

    public static void delete(String key) {
        UserAttributesDbHelper.delete(key, UserManager.getUUID());
    }

    public static void deleteAll(@UserAttribute.Type int type) {
        UserAttributesDbHelper.deleteType(UserManager.getUUID(), type);
    }

    public static void deleteAnonymousUserAttribute() {
        UserAttributesDbHelper.deleteAnonymousData();
    }

    public static void insertAll(List<UserAttribute> userAttributeList) {
        UserAttributesDbHelper.insertBulk(userAttributeList);
    }

    @NonNull
    public static String getSDKUserAttributesAndAppendToIt(@NonNull Map<String, String> attributes) {
        return UserAttributesDbHelper.getSDKUserAttributesAndAppendToIt(attributes);
    }

    public static void deleteAll() {
        UserAttributesDbHelper.deleteUserAttributes(UserManager.getUUID());
    }
}
