package com.instabug.library.internal.storage;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;
import android.provider.OpenableColumns;
import android.util.Pair;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.model.Attachment;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;

public class AttachmentsUtility {
    public static final double MAX_FILE_SIZE_IN_MB = 5.0D;

    public static File getVideoFile(Context context) {
        File videoDirectory = getNewDirectory(context, "videos");
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss.SSS", Locale.ENGLISH);

        String videoName = "video-" + dateFormat.format(new Date()) + ".mp4";
        return new File(videoDirectory, videoName);
    }

    public static File getFilesAttachmentDirectory(Context context) {
        return getNewDirectory(context, "attachments");
    }

    public static File getNewDirectory(Context context, String directoryName) {
        File directory = new File(DiskUtils.getInstabugInternalDirectory(context) + "/" + directoryName +
                "/");
        if (!directory.exists()) {
            if (directory.mkdirs()) {
                File noMediaFile = new File(directory, ".nomedia");
                try {
                    noMediaFile.createNewFile();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return directory;
    }

    /**
     * A method used by {BaseReportingPresenter} & {ChatPresenter}
     * This should return the display name with file extension of the file / video and size
     *
     * @return pair of Strings first is the name, second is the size
     */
    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static Pair<String, String> getFileNameAndSize(Context context, @Nullable Uri uri) {
        if (uri == null) {
            return null;
        }
        Cursor cursor = context.getContentResolver()
                .query(uri, null, null, null, null);
        try {
            if (cursor != null && cursor.moveToFirst()) {
                String name = cursor.getString(
                        cursor.getColumnIndexOrThrow(OpenableColumns.DISPLAY_NAME));
                String size = cursor.getString(
                        cursor.getColumnIndexOrThrow(OpenableColumns.SIZE));

                return new Pair<>(name, size);
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return null;
    }


    @Nullable
    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    public static InputStream getFileInputStreamFileFromUri(Context context, Uri fileUri) {
        try {
            return context.getContentResolver().openInputStream(fileUri);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static String getGalleryImagePath(Activity activity, @Nullable Uri uri) {
        if (uri == null) {
            return null;
        }
        String[] projection = {MediaStore.Images.Media.DATA};
        Cursor cursor = activity.managedQuery(uri, projection, null, null, null);
        if (cursor != null) {
            int column_index = cursor
                    .getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
            cursor.moveToFirst();
            return cursor.getString(column_index);
        } else {
            return null;
        }
    }

    /**
     * Returns a local file copy of the file at the specified URI {@code originalUri}
     *
     * @param context     to get attachment directory
     * @param originalUri of the original file to be copied
     * @return the new copied file
     */
    @Nullable
    public static Uri getNewFileAttachmentUri(@Nullable Context context, Uri originalUri,
                                              @Nullable String fileNameWithExtension) {
        if (originalUri == null || context == null || originalUri.getPath() == null)
            return null;

        String lastSegment = originalUri.getLastPathSegment();
        File attachmentDirectory = getFilesAttachmentDirectory(context);
        String fileName = lastSegment == null ? "" : lastSegment.toLowerCase();

        if (fileNameWithExtension != null && SettingsManager.getInstance().getExtraAttachmentFiles() != null &&
                SettingsManager.getInstance().getExtraAttachmentFiles().containsKey(originalUri)) {
            fileName = fileNameWithExtension;
        }

        File attachmentFile = new File(attachmentDirectory, fileName);
        Uri attachmentFileUri;
        if (attachmentFile.exists()) {
            attachmentFile = new File(attachmentDirectory,
                    System.currentTimeMillis() + "_" + fileName);
        }

        try {
            if (fileSizeIsNotValidWithLog(originalUri, MAX_FILE_SIZE_IN_MB)) {
                return null;
            }
            DiskUtils.copyFromUriIntoFile(context, originalUri, attachmentFile);
            attachmentFileUri = Uri.fromFile(attachmentFile);
            if (fileSizeIsNotValidWithLog(attachmentFileUri, MAX_FILE_SIZE_IN_MB)) {
                return null;
            }
        } catch (IOException ex) {
            String message = ex.getMessage();
            if (message == null) {
                message = "Exception while copying attachment file";
            }
            InstabugSDKLogger.e(Constants.LOG_TAG, message, ex);
            return null;
        }

        return attachmentFileUri;
    }

    /**
     * Returns a local file copy of the file at the specified URI With specific file size limit{@code originalUri}
     *
     * @param context            to get attachment directory
     * @param originalUri        of the original file to be copied
     * @param fileSizeLimitInMbs File size limit in megabytes
     * @return the new copied file
     */
    @Nullable
    public static Uri getNewFileAttachmentUri(@Nullable Context context, Uri originalUri,
                                              @Nullable String fileNameWithExtension, double fileSizeLimitInMbs) {
        if (originalUri == null || context == null || originalUri.getPath() == null)
            return null;

        String lastSegment = originalUri.getLastPathSegment();
        File attachmentDirectory = getFilesAttachmentDirectory(context);
        String fileName = lastSegment == null ? "" : lastSegment.toLowerCase();

        if (fileNameWithExtension != null && SettingsManager.getInstance().getExtraAttachmentFiles() != null &&
                SettingsManager.getInstance().getExtraAttachmentFiles().containsKey(originalUri)) {
            fileName = fileNameWithExtension;
        }

        File attachmentFile = new File(attachmentDirectory, fileName);
        Uri attachmentFileUri;
        if (attachmentFile.exists()) {
            attachmentFile = new File(attachmentDirectory,
                    System.currentTimeMillis() + "_" + fileName);
        }

        try {
            if (fileSizeIsNotValidWithLog(originalUri, fileSizeLimitInMbs)) {
                return null;
            }
            DiskUtils.copyFromUriIntoFile(context, originalUri, attachmentFile);
            attachmentFileUri = Uri.fromFile(attachmentFile);
            if (fileSizeIsNotValidWithLog(attachmentFileUri, fileSizeLimitInMbs)) {
                return null;
            }
        } catch (IOException ex) {
            String message = ex.getMessage();
            if (message == null) {
                message = "Exception while copying attachment file";
            }
            InstabugSDKLogger.e(Constants.LOG_TAG, message, ex);
            return null;
        }
        return attachmentFileUri;
    }

    private static boolean fileSizeIsNotValidWithLog(Uri fileUri, double maxFileSizeInMb) {
        if (!validateFileSize(fileUri, maxFileSizeInMb)) {
            InstabugSDKLogger.w(Constants.LOG_TAG,
                    "Attachment file " + fileUri.toString() + " size exceeds than the limit " + maxFileSizeInMb);
            return true;
        }
        return false;
    }

    @Nullable
    public static Uri getNewFileAttachmentUri(@Nullable Context context, Uri originalUri) {
        return getNewFileAttachmentUri(context, originalUri, null);
    }

    @Nullable
    public static Uri getUriFromBytes(Context context, byte[] data, String fileName) {
        File attachmentFile = getInternalAttachmentFile(context, fileName);

        try {
            saveBytesToFile(data, attachmentFile);
            boolean isEncrypted = FileUtils.encryptFile(attachmentFile.getPath());
            if (isEncrypted) {
                attachmentFile = new File(FileUtils.getPathWithEncryptedFlag(attachmentFile.getPath()));
            }

        } catch (IOException ex) {
            if (ex.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting uri: " + ex.getMessage(), ex);
            }
            return null;
        }

        return Uri.fromFile(attachmentFile);
    }

    private static File getInternalAttachmentFile(Context context, String fileName) {
        File attachmentDirectory = getFilesInternalAttachmentDirectory(context);

        File attachmentFile = new File(attachmentDirectory, fileName);
        if (attachmentFile.exists()) {
            attachmentFile = new File(attachmentDirectory, System.currentTimeMillis()
                    + "_" + fileName);
        }

        return attachmentFile;
    }

    public static File getFilesInternalAttachmentDirectory(Context context) {
        return getNewDirectory(context, "internal-attachments");
    }

    @SuppressLint("RESOURCE_LEAK")
    private static void saveBytesToFile(byte[] data, File file) throws IOException {
        BufferedOutputStream bos = null;
        try {
            bos = new BufferedOutputStream(new FileOutputStream(file));
            bos.write(data);
        } finally {
            if (bos != null) {
                bos.flush();
                bos.close();
            }
        }
    }

    /**
     * A method to save input stream to a file
     *
     * @param context     android context
     * @param inputStream InputStream to read data from
     * @param fileName    The new file name
     * @return a new File with the given fileName
     */
    @Nullable
    @SuppressLint("RESOURCE_LEAK")
    public static File saveInputStreamToFile(Context context, InputStream inputStream, String fileName) {
        File newFile = new File(getFilesAttachmentDirectory(context), fileName);
        FileOutputStream os = null;
        try {
            os = new FileOutputStream(newFile);
            byte[] buffer = new byte[1024];
            int bytesRead;
            //read from is to buffer
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
            inputStream.close();
            return newFile;
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "IO exception: " + e.getMessage(), e);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.toString(), e);
        } finally {
            try {
                if (os != null) {
                    os.flush();
                    os.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    /**
     * A method to copy file from content provider to our library
     *
     * @param context  Android Context
     * @param uri      URI from content provider
     * @param fileName the desired file name
     * @return a new file with the selected file name
     */
    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    @Nullable
    public static File getFileFromContentProvider(Context context, @Nullable Uri uri, String fileName) {
        if (uri == null) {
            return null;
        }
        InputStream inputStream = getFileInputStreamFileFromUri(context, uri);
        if (inputStream != null) {
            return AttachmentsUtility.saveInputStreamToFile(context, inputStream, fileName);
        }
        InstabugSDKLogger.e(Constants.LOG_TAG, "Inputstream is null while reading file from content provider");
        return null;
    }

    /**
     * Validate the file size.
     *
     * @param originalUri original file URI
     * @param maxFileSize in mega byte
     * @return true if the file size is lower than the maxFileSize
     */
    public static boolean validateFileSize(Uri originalUri, double maxFileSize) {
        if (Instabug.getApplicationContext() == null) return false;
        if (originalUri.getPath() == null) return false;

        long sizeInBytes;
        if (originalUri.toString().contains(ContentResolver.SCHEME_ANDROID_RESOURCE)) {

            InputStream inputStream = null;
            try {
                inputStream = Instabug.getApplicationContext().getContentResolver().openInputStream(originalUri);
                if (inputStream == null) {
                    return false;
                }

                sizeInBytes = inputStream.available();
            } catch (IOException e) {
                InstabugSDKLogger.w(Constants.LOG_TAG, "External attachment file " + originalUri.getPath()
                        + " couldn't be loaded to calculate its size");
                return false;
            } finally {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (IOException e) {
                        //Ignore exception
                    }
                }
            }
        } else {
            File file = new File(originalUri.getPath());
            sizeInBytes = file.length();
        }

        double sizeInMb = sizeInBytes / ((double) 1024 * 1024);
        if (sizeInMb > maxFileSize) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "External attachment file size is " +
                    sizeInBytes + " bytes or " + sizeInMb + " MBs > maxFileSize " + maxFileSize);
            return false;
        }
        return isValidSize(sizeInBytes, maxFileSize);
    }

    public static boolean isValidSize(double sizeInBytes, double maxFileSize) {
        double sizeInMb = sizeInBytes / ((double) 1024 * 1024);
        if (sizeInMb > maxFileSize) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "External attachment file size is " + sizeInMb + " MBs > maxFileSize " + maxFileSize);
            return false;
        }
        return true;
    }

    /**
     * A method that encrypts the attachments
     *
     * @param attachments list of attachments
     */
    public static void encryptAttachments(@Nullable List<Attachment> attachments) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "encryptAttachments");
        // encrypting attachments
        if (attachments != null) {
            for (Attachment attachment : attachments) {
                if (!attachment.isEncrypted() && attachment.getLocalPath() != null) {
                    boolean isFileEncrypted = InstabugCore.encrypt(attachment.getLocalPath());
                    attachment.setEncrypted(isFileEncrypted);
                }
            }
        }
    }

    /**
     * A helper method to encrypt
     *
     * @param attachments
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static void encryptAttachmentsAndUpdateDb(@Nullable List<Attachment> attachments) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "encryptAttachmentsAndUpdateDb");
        if (attachments != null) {
            for (Attachment attachment : attachments) {
                encryptAttachmentAndUpdateDb(attachment);
            }
        }
    }

    /**
     * A helper method to encrypt
     *
     * @param attachment
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static void encryptAttachmentAndUpdateDb(Attachment attachment) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "encryptAttachmentAndUpdateDb");
        if (!attachment.isEncrypted() && attachment.getLocalPath() != null) {
            boolean isAttachmentEncrypted = InstabugCore.encrypt(attachment.getLocalPath());
            attachment.setEncrypted(isAttachmentEncrypted);
            ContentValues cv = new ContentValues();
            cv.put(InstabugDbContract.AttachmentEntry.COLUMN_ENCRYPTED, isAttachmentEncrypted);
            AttachmentsDbHelper.update(attachment.getId(), cv);
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static boolean decryptAttachmentAndUpdateDb(Attachment attachment) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "decryptAttachmentAndUpdateDb");
        if (attachment.isEncrypted() && attachment.getLocalPath() != null) {
            boolean isAttachmentDecrypted = InstabugCore.decrypt(attachment.getLocalPath());
            attachment.setEncrypted(!isAttachmentDecrypted);
            ContentValues cv = new ContentValues();
            if (isAttachmentDecrypted) {
                String decryptedFilePath = FileUtils.getPathWithDecryptedFlag(attachment.getLocalPath());
                cv.put(InstabugDbContract.AttachmentEntry.COLUMN_LOCALE_PATH,
                        decryptedFilePath);
                attachment.setLocalPath(decryptedFilePath);
                String decryptedFileName = FileUtils.getFileName(attachment.getLocalPath());
                if (decryptedFileName != null)
                    attachment.setName(decryptedFileName);
            }
            cv.put(InstabugDbContract.AttachmentEntry.COLUMN_ENCRYPTED, !isAttachmentDecrypted);
            AttachmentsDbHelper.update(attachment.getId(), cv);
            return isAttachmentDecrypted;
        }
        // The attachment is already decrypted
        return true;
    }

    public static void clearInternalAttachments(@Nullable Context context) {
        if (context != null) {
            File file = getFilesInternalAttachmentDirectory(context);
            FileUtils.deleteDirectory(file);
        }
    }
}
