package com.instabug.library.internal.dataretention.files;

import com.instabug.library.Constants;
import com.instabug.library.internal.dataretention.core.Scope;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import androidx.annotation.NonNull;

public abstract class FileScope implements Scope<FileRecord> {

    @NonNull
    @Override
    public abstract Collection<FileRecord> records();

    /**
     * @return total size in bytes or zero if the directory does not exist or an error has occurred
     */
    @Override
    public abstract long size();

    public static class Factory {

        @NonNull
        public FileScope create(@NonNull final String path) {
            return new FileScope() {

                @NonNull
                @Override
                public Collection<FileRecord> records() {
                    return queryAllRecords(path);
                }

                @Override
                public long size() {
                    return calculateSizeOnDisk(path);
                }
            };
        }

        @NonNull
        protected Collection<FileRecord> queryAllRecords(@NonNull String path) {
            final List<FileRecord> records = new LinkedList<>();
            try {
                File directory = new File(path);
                File[] files = directory.listFiles();
                if (directory.exists() && files != null) {
                    for (File file : files) {
                        records.add(createRecord(file));
                    }
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting files to delete", e);
            }
            return records;
        }

        protected long calculateSizeOnDisk(@NonNull String path) {
            try {
                return FileUtils.getSize(new File(path));
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting file size", e);
                return 0;
            }
        }

        @NonNull
        protected FileRecord createRecord(@NonNull File file) {
            return new FileRecord(file);
        }
    }
}
