package com.instabug.library.internal.data;

import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;

import java.util.List;

import io.reactivexport.Observable;

/**
 * A representation of any data source. This interface should be typically implemented in the model layer
 *
 * @param <T> The type of actual data that can be queried, save or deleted
 * @param <V> The type of data identifier
 */
public abstract class DataSource<T, V> {

    /**
     * Adds the specified element to the data source or updates it if it already exists
     * <p>
     * Data sources that support this operation may place limitations on what
     * elements may be added to this data source.  In particular, some
     * data sources will refuse to add null elements, and others will impose
     * restrictions on the type of elements that may be added.  Data source
     * classes should clearly specify in their documentation any restrictions
     * on what elements may be added.
     *
     * @param entity element to be appended to or updated in this data source
     */
    public void saveOrUpdate(T entity, Request.Callbacks<RequestResponse, Throwable> callback) {
        throw new UnsupportedOperationException();
    }

    /**
     * Adds a bulk of specified elements to the data source or updates whatever already exists
     * <p>
     * Data sources that support this operation may place limitations on what
     * elements may be added to this data source.  In particular, some
     * data sources will refuse to add null elements, and others will impose
     * restrictions on the type of elements that may be added.  Data source
     * classes should clearly specify in their documentation any restrictions
     * on what elements may be added.
     *
     * @param entities elements to be appended to or updated in this data source
     */
    public void saveOrUpdateBatch(List<T> entities) {
        throw new UnsupportedOperationException();
    }

    /**
     * Queries a specific data record in this data source given its {@code identifier}
     * <p>
     * Data sources that support this operation may place limitations on what
     * elements may be added to this data source.  In particular, some
     * data sources will refuse to add null elements, and others will impose
     * restrictions on the type of elements that may be added.  Data source
     * classes should clearly specify in their documentation any restrictions
     * on what elements may be added.
     *
     * @param identifier the queried data identifier
     * @return An {@link Observable} of the queried data if found
     * or {@link Observable#empty()}otherwise
     */
    public T query(V identifier) {
        throw new UnsupportedOperationException();
    }

    /**
     * Queries all data records in this data source given its {@code identifier}
     * <p>
     * Data sources that support this operation may place limitations on what
     * elements may be added to this data source.  In particular, some
     * data sources will refuse to add null elements, and others will impose
     * restrictions on the type of elements that may be added.  Data source
     * classes should clearly specify in their documentation any restrictions
     * on what elements may be added.
     *
     * @return An {@link Observable} of the queried data if found
     * or {@link Observable#empty()}otherwise
     */
    public List<T> queryAll() {
        throw new UnsupportedOperationException();
    }

    /**
     * Deletes the occurrences of the specified element from this data source,
     * if it is present (optional operation).  If this data source does not contain
     * the element, it is unchanged.
     *
     */
    public void delete(T entity) {
        throw new UnsupportedOperationException();
    }

    /**
     * Clears this data source,
     *
     */
    public void deleteAll() {
        throw new UnsupportedOperationException();
    }
}
