package com.instabug.library.encryption

import android.os.Build
import android.util.Base64
import androidx.annotation.IntDef
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.encryption.iv.IVManager
import com.instabug.library.util.InstabugSDKLogger
import java.nio.charset.Charset
import java.security.spec.AlgorithmParameterSpec
import javax.crypto.Cipher
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.IvParameterSpec

object EncryptionManager {

    @IntDef(IV_V1, IV_V2)
    @Retention(AnnotationRetention.SOURCE)
    annotation class IvVersion
    const val IV_V1 = 1
    const val IV_V2 = 2

    private const val AES_MODE = "AES/GCM/NoPadding"
    private const val ENCRYPTION_PREFIX = "^instaEncrypted^"
    private const val ENCRYPTION_NEW_LINE_REPLACEMENT = "^instaLINE^"
    private const val IV_LENGTH = 128

    const val LINE_FEED = "\n\r"
    val iv = "RandomAESIv1".toByteArray()


    @JvmStatic
    fun encryptWithStaticKey(data: String?) : String? {
        return if (data != null) {
            val cipher: Cipher?
            try {
                if (data.startsWith(ENCRYPTION_PREFIX)) {
                    return data
                }
                cipher = Cipher.getInstance(AES_MODE)
                val ivSpec = getIvSpec()
                cipher.init(Cipher.ENCRYPT_MODE, StaticKeyProvider.getStaticKey(), ivSpec)
                val cipherText: ByteArray = cipher.doFinal(data.toByteArray())
                return Base64.encodeToString(cipherText, Base64.DEFAULT)
                        .replace("\n", ENCRYPTION_NEW_LINE_REPLACEMENT)
            } catch (e: Exception) {
                IBGDiagnostics.reportNonFatalAndLog(
                    e,
                    "Error while encrypting string, returning original string",
                    Constants.LOG_TAG
                )
                data
            } catch (oom: OutOfMemoryError) {
                IBGDiagnostics.reportNonFatalAndLog(
                    oom,
                    "OOM while encrypting string, returning original string",
                    Constants.LOG_TAG
                )
                data
            }
        } else {
            null
        }
    }

    @JvmStatic
    fun decryptWithStaticKey(data: String?): String? {
        return if (data != null) {
            if (data.isEmpty()) {
                return ""
            }


            val encryptedData = data.replace(ENCRYPTION_NEW_LINE_REPLACEMENT, "\n")

            val encryptedBytes: ByteArray = try {
                Base64.decode(
                    encryptedData,
                    Base64.DEFAULT
                )
            } catch (exception: IllegalArgumentException) {
                return encryptedData
            }
            try {
                val cipher = Cipher.getInstance(AES_MODE)
                val ivSpec = getIvSpec ()
                cipher.init(Cipher.DECRYPT_MODE, StaticKeyProvider.getStaticKey(), ivSpec)
                val decryptedBytes = cipher.doFinal(encryptedBytes)
                return String(decryptedBytes, Charset.forName("UTF-8"))
            } catch (e: Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while decrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    e,
                    "Error: " + e.message + "while decrypting string, returning original string"
                )
                data
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "OOM while decrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    oom,
                    "OOM while decrypting string, returning original string"
                )
                data
            }
        } else {
            null
        }
    }

    @JvmStatic
    fun encrypt(data: String?): String? {
        return encrypt(data, IV_V1)
    }

    @JvmStatic
    fun encrypt(data: String?, @IvVersion ivVersion: Int): String? {
        return if (data != null) {
            val cipher: Cipher?
            try {
                if (data.startsWith(ENCRYPTION_PREFIX)) {
                    return data
                }
                cipher = Cipher.getInstance(AES_MODE)
                val ivSpec = if (ivVersion == IV_V1) getIvSpec() else getIvSpecV2()
                cipher.init(Cipher.ENCRYPT_MODE, KeyManager.getKey(), ivSpec)
                val cipherText: ByteArray = cipher.doFinal(data.toByteArray())
                return "$ENCRYPTION_PREFIX${
                    Base64.encodeToString(cipherText, Base64.DEFAULT)
                        .replace("\n", ENCRYPTION_NEW_LINE_REPLACEMENT)
                }"
            } catch (e: Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while encrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    e,
                    "Error: " + e.message + "while encrypting string, returning original string"
                )
                data
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "OOM while encrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    oom,
                    "OOM while encrypting string, returning original string"
                )
                data
            }
        } else {
            null
        }
    }

    @JvmStatic
    fun decrypt(data: String?): String? {
        return decrypt(data, IV_V1)
    }

    @JvmStatic
    fun decrypt(data: String?, @IvVersion ivVersion: Int): String? {
        return if (data != null) {
            if (data.isEmpty()) {
                return ""
            }
            if (!data.startsWith(ENCRYPTION_PREFIX)) {
                return data
            }
            val encryptedData = data.substring(ENCRYPTION_PREFIX.length, data.length)

            val encryptedBytes: ByteArray = try {
                Base64.decode(
                    encryptedData.replace(ENCRYPTION_NEW_LINE_REPLACEMENT, "\n"),
                    Base64.DEFAULT
                )
            } catch (exception: IllegalArgumentException) {
                return encryptedData
            }
            try {
                val cipher = Cipher.getInstance(AES_MODE)
                val ivSpec = if (ivVersion == IV_V1) getIvSpec() else getIvSpecV2()
                cipher.init(Cipher.DECRYPT_MODE, KeyManager.getKey(), ivSpec)
                val decryptedBytes = cipher.doFinal(encryptedBytes)
                return String(decryptedBytes, Charset.forName("UTF-8"))
            } catch (e: Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Error while decrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    e,
                    "Error: " + e.message + "while decrypting string, returning original string"
                )
                data
            } catch (oom: OutOfMemoryError) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "OOM while decrypting string, returning original string"
                )
                IBGDiagnostics.reportNonFatal(
                    oom,
                    "OOM while decrypting string, returning original string"
                )
                data
            }
        } else {
            null
        }
    }

    @JvmStatic
    @Throws(Exception::class, OutOfMemoryError::class)
    fun encrypt(data: ByteArray): ByteArray {
        return try {
            val cipher = Cipher.getInstance(AES_MODE)
            cipher.init(Cipher.ENCRYPT_MODE, KeyManager.getKey(), getIvSpec())
            cipher.doFinal(data)
        } catch (e: Exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while encrypting bytes")
            data
        }
    }

    @JvmStatic
    @Throws(Exception::class, OutOfMemoryError::class)
    fun decrypt(data: ByteArray): ByteArray {
        return try {
            val cipher = Cipher.getInstance(AES_MODE)
            cipher.init(Cipher.DECRYPT_MODE, KeyManager.getKey(), getIvSpec())
            return cipher.doFinal(data)
        } catch (e: Exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while decrypting bytes")
            data
        }
    }

    private fun getIvSpec(): AlgorithmParameterSpec {
        return if (Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            GCMParameterSpec(96, iv)
        } else {
            IvParameterSpec(iv)
        }
    }

    @Synchronized
    private fun getIvSpecV2(): AlgorithmParameterSpec {
        val iv = IVManager.getIV()
        return if (Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            GCMParameterSpec(IV_LENGTH, iv)
        } else {
            IvParameterSpec(iv)
        }
    }
}
