package com.instabug.library.diagnostics.nonfatals.settings;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.settings.SettingsManager;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;
import java.util.Set;

/**
 * Created by Shannan on 28/08/2021.
 */
public class NonFatalSettings {


    private static final String ENABLED_KEY = "enabled";
    private static final String NON_FATALS_MAX_COUNT_KEY = "non_fatals_max_count";
    private static final String OCCURRENCES_MAX_COUNT_KEY = "occurrences_max_count";
    private static final String IGNORE_LIST_KEY = "blacklist";
    private static final String ANDROID_KEY = "android";
    private static final String EXCEPTION_TYPE_KEY = "exception_type";
    private static final String CLASS_KEY = "class";
    private static final String METHOD_NAME_KEY = "method_name";
    private static final String FILE_NAME_KEY = "file_name";

    @Nullable
    private static NonFatalSettings nonFatalSettings;
    private boolean enabled;
    private int nonFatalsMaxCount;
    private int occurrencesMaxCount;
    @Nullable
    private Set<NonFatal> ignoreList;

    public NonFatalSettings() {
    }

    public synchronized static void init() {
        nonFatalSettings = new NonFatalSettings();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return a {@code BugSettings} instance
     */
    public synchronized static NonFatalSettings getInstance() {
        if (nonFatalSettings == null) {
            nonFatalSettings = new NonFatalSettings();
        }
        return nonFatalSettings;
    }

    public synchronized static void release() {
        nonFatalSettings = null;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public int getNonFatalsMaxCount() {
        return nonFatalsMaxCount;
    }

    public void setNonFatalsMaxCount(int nonFatalsMaxCount) {
        this.nonFatalsMaxCount = nonFatalsMaxCount;
    }

    public int getOccurrencesMaxCount() {
        return occurrencesMaxCount;
    }

    public void setOccurrencesMaxCount(int occurrencesMaxCount) {
        this.occurrencesMaxCount = occurrencesMaxCount;
    }

    @Nullable
    public Set<NonFatal> getIgnoreList() {
        return ignoreList;
    }

    public void setIgnoreList(@Nullable Set<NonFatal> ignoreList) {
        this.ignoreList = ignoreList;
    }

    public void fromJson(String modelAsJson) throws JSONException {
        fromJson(new JSONObject(modelAsJson));
    }

    public void fromJson(@NonNull JSONObject jsonObject) throws JSONException {
        setEnabled(SettingsManager.getInstance().getFeatureState(IBGFeature.NON_FATAL_ERRORS, DefaultValues.IS_NON_FATAL_FEATURE_AVAILABLE) == Feature.State.ENABLED);
        setNonFatalsMaxCount(jsonObject.optInt(NON_FATALS_MAX_COUNT_KEY, DefaultValues.NON_FATALS_MAX_COUNT));
        setOccurrencesMaxCount(jsonObject.optInt(OCCURRENCES_MAX_COUNT_KEY, DefaultValues.OCCURRENCES_MAX_COUNT));
        if (jsonObject.has(IGNORE_LIST_KEY)) {
            try {
                JSONObject ignoreListObject = jsonObject.getJSONObject(IGNORE_LIST_KEY);
                setIgnoreList(fillIgnoreListSet(ignoreListObject));
            } catch (JSONException e) {
                setIgnoreList(null);
            }
        } else {
            setIgnoreList(null);
        }
    }

    public Set<NonFatal> fillIgnoreListSet(@Nullable JSONObject ignoreListJsonObj) throws JSONException {

        HashSet<NonFatal> targetSet = new HashSet<>();
        if (ignoreListJsonObj == null || !ignoreListJsonObj.has(ANDROID_KEY))
            return targetSet;

        JSONArray androidBL = ignoreListJsonObj.getJSONArray(ANDROID_KEY);
        NonFatal nonFatal;
        for (int i = 0; i < androidBL.length(); i++) {
            JSONObject object = androidBL.getJSONObject(i);
            nonFatal = new NonFatal();
            nonFatal.setExceptionType(object.getString(EXCEPTION_TYPE_KEY));
            nonFatal.setDeclaringClass(object.getString(CLASS_KEY));
            nonFatal.setFileName(object.getString(FILE_NAME_KEY));
            nonFatal.setMethodName(object.getString(METHOD_NAME_KEY));
            targetSet.add(nonFatal);
        }
        return targetSet;
    }
}