package com.instabug.library.diagnostics.customtraces.cache

import androidx.annotation.VisibleForTesting
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.customtraces.model.IBGCustomTrace
import com.instabug.library.diagnostics.customtraces.settings.CustomTracesSettings
import com.instabug.library.diagnostics.customtraces.settings.CustomTracesSettingsResolver

class CustomTracesCacheManagerImpl constructor(
    @get:VisibleForTesting
    val tracesDBHelper: CustomTracesDBHelper = CustomTracesServiceLocator.getTracesDBHelper(),
    @get:VisibleForTesting
    val attributesDBHelper: AttributesDBHelper = CustomTracesServiceLocator.getAttributesDBHelper()
) : CustomTracesCacheManager {

    @get:VisibleForTesting
    val tracesSettings: CustomTracesSettings
        get() = CustomTracesSettingsResolver.customTracesSettings ?: CustomTracesSettings()

    override fun startTrace(
        trace: IBGCustomTrace
    ): Long {
        return tracesDBHelper.insertTrace(trace)
            .takeIf { it != -1L }
            ?.also {
                tracesDBHelper.trimToLimit(tracesSettings.maxCount)
            } ?: -1
    }

    override fun endTrace(traceId: Long, duration: Long, endedInBG: Boolean): Boolean {
        return tracesDBHelper.endTrace(traceId, duration, endedInBG)
    }

    override fun setAttribute(traceId: Long, key: String, value: String?): Boolean {
        return value?.let {
            attributesDBHelper.addAttribute(traceId, key, it)
        } ?: false

    }

    override fun updateAttribute(traceId: Long, key: String, value: String?): Boolean {
        return if (value == null)
            attributesDBHelper.removeAttribute(traceId, key)
        else
            attributesDBHelper.updateAttribute(traceId, key, value)
    }

    override fun getAllTraces(): List<IBGCustomTrace> {
        return tracesDBHelper.getAllTraces().onEach { trace ->
            trace.setAttributesWithoutCaching(attributesDBHelper.getTraceAttributes(trace.id))
        }
    }

    override fun removeUnEndedTraces() {
        tracesDBHelper.removeUnEndedTraces()
    }

    override fun deleteAll() {
        tracesDBHelper.removeAll()
    }

    override fun logTrace(traceName: String, startTime: Long, duration: Long, isBG: Boolean) {
        tracesDBHelper.insertTrace(
            IBGCustomTrace(
                name = traceName,
                startTime = startTime,
                duration = duration,
                startedInBG = isBG,
                endedInBG = isBG
            )
        ).takeIf { it != -1L }?.also {
            tracesDBHelper.trimToLimit(tracesSettings.maxCount)
        }
    }

    override fun getTraceId(trace: IBGCustomTrace): Long =
        tracesDBHelper.getTraceId(trace)


    override fun clearSyncedTraces(traces: List<IBGCustomTrace>) {
        val ids = mutableListOf<Long>()
        traces.forEach { trace ->
            getTraceId(trace).takeIf { id ->
                id != -1L
            }?.let { id ->
                ids.add(id)
            }
        }

        takeIf { ids.isNotEmpty() }?.let {
            tracesDBHelper.deleteTraces(ids)
        }
    }

    override fun clearTracesByName(tracesNames: Array<String>) {
        tracesDBHelper.deleteTracesByName(tracesNames.toList())
    }
}