package com.instabug.library.diagnostics.customtraces

import androidx.annotation.WorkerThread
import com.instabug.library.diagnostics.DiagnosticsConstants
import com.instabug.library.diagnostics.customtraces.cache.CustomTracesCacheManager
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.customtraces.model.IBGCustomTrace
import com.instabug.library.diagnostics.customtraces.utils.CustomTracesValidator
import com.instabug.library.diagnostics.customtraces.utils.CustomTracesValidator.toValidTraceNameOrNull
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.threading.executeAndGet
import java.util.concurrent.ThreadPoolExecutor

class CustomTracesManagerImpl(
    private val cacheManager: CustomTracesCacheManager = CustomTracesServiceLocator.getCacheManager(),
    private val executor: ThreadPoolExecutor = CustomTracesServiceLocator.getCustomTracesExecutor()
) : CustomTracesManager {

    private val lock: Any = Any()

    override fun clearCache() {
        executor.execute {
            synchronized(lock) {
                cacheManager.deleteAll()
            }
        }
    }

    override fun removeUnEndedTraces() {
        executor.execute {
            synchronized(lock) {
                cacheManager.removeUnEndedTraces()
            }
        }
    }

    override fun startTrace(traceName: String?, startTime: Long): IBGCustomTrace? {
        val stackTrace = Thread.currentThread().stackTrace
        return executor.executeAndGet<IBGCustomTrace> {
            synchronized(lock) {
                let {
                    CustomTracesValidator.canStartTrace(stackTrace)
                }.takeIf {
                    it
                }?.let {
                    traceName.toValidTraceNameOrNull()
                }?.run {
                    val startedInBG =
                        InstabugInternalTrackingDelegate.getInstance().startedActivitiesNumber <= 0
                    val trace = IBGCustomTrace(
                        name = this,
                        startTimeMicros = System.nanoTime() / 1000,
                        startTime = startTime,
                        startedInBG = startedInBG
                    )
                    val traceId = cacheManager.startTrace(trace)
                    if (traceId != -1L) {
                        trace.id = traceId
                        return@executeAndGet trace
                    }

                }

                return@executeAndGet null
            }
        }
    }


    override fun logTrace(traceName: String?, startTime: Long, endTime: Long) {
        val stackTrace = Thread.currentThread().stackTrace
        executor.execute {
            synchronized(lock) {
                takeIf {
                    CustomTracesValidator.canStartTrace(stackTrace) && CustomTracesValidator.validateTraceDuration(
                        startTime,
                        endTime
                    )
                }?.let {
                    traceName.toValidTraceNameOrNull()
                }?.run {
                    val isBG =
                        InstabugInternalTrackingDelegate.getInstance().startedActivitiesNumber <= 0
                    cacheManager.logTrace(
                        traceName = this,
                        startTime = startTime,
                        duration = endTime - startTime,
                        isBG = isBG
                    )
                }
            }
        }
    }

    override fun updateAttribute(traceId: Long, key: String, value: String?): Boolean? {
        return executor.executeAndGet {
            synchronized(lock) {
                cacheManager.updateAttribute(traceId, key, value)
            }
        }
    }

    override fun setAttribute(traceId: Long, key: String, value: String?): Boolean? {
        return executor.executeAndGet() {
            synchronized(lock) {
                cacheManager.setAttribute(traceId, key, value)
            }
        }
    }

    override fun endTrace(traceId: Long, duration: Long, endedInBG: Boolean): Boolean? {
        return executor.executeAndGet {
            synchronized(lock) {
                cacheManager.endTrace(traceId, duration, endedInBG)
            }
        }
    }

    @WorkerThread
    override fun getAllTraces(): List<IBGCustomTrace> {
        return synchronized(lock) {
            cacheManager.getAllTraces()
        }
    }

    override fun clearSyncedTraces(traces: List<IBGCustomTrace>?) {
        executor.execute {
            synchronized(lock) {
                traces?.let {
                    cacheManager.clearSyncedTraces(it)
                }
            }
        }
    }

    override fun clearTracesForFlag(flagName: String) {
        executor.execute {
            synchronized(lock) {
                val tracesNames = when(flagName) {
                    DiagnosticsConstants.RECORD_LAUNCH_TRACE -> DiagnosticsConstants.LAUNCH_TRACES
                    DiagnosticsConstants.RECORD_FEATURE_TRACE -> DiagnosticsConstants.FEATURE_TRACES
                    else -> arrayOf()
                }

                tracesNames.takeIf {
                    it.isNotEmpty()
                }?.let {
                    cacheManager.clearTracesByName(it)
                }
            }
        }
    }
}