package com.instabug.library.crash;


import static com.instabug.library.sessionV3.sync.SessionBatchingFilterKt.getNoneFilter;

import android.content.Context;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.SessionManager;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.model.v3Session.SessionEvent;
import com.instabug.library.session.SessionsLocalDataSource;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessionV3.manager.IBGSessionManager;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.PrintWriter;
import java.io.StringWriter;

public class InstabugUncaughtExceptionHandler implements Thread.UncaughtExceptionHandler {

    @Nullable
    private final Thread.UncaughtExceptionHandler defaultUncaughtExceptionHandler;

    public InstabugUncaughtExceptionHandler() {
        defaultUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    }

    @Override
    public void uncaughtException(final Thread thread, final Throwable throwable) {
        InstabugSDKLogger.e(Constants.LOG_TAG, "Instabug Caught an Unhandled Exception: " +
                throwable.getClass().getCanonicalName(), throwable);
        try {
            if (InternalAutoScreenRecorderHelper.getInstance().isEnabled()) {
                InternalAutoScreenRecorderHelper.getInstance().setCrashOccurred(true);
            }

            SettingsManager.getInstance().setCrashedSession(true);

            // finish the current session if only a native crash
            if (!isReactNativeCrash(getStackTrace(throwable))) {
                SessionManager.getInstance().finishSession();
                IBGSessionManager.INSTANCE.emit(new SessionEvent.End(true,false), true);
            }

            Context context = Instabug.getApplicationContext();
            if (context != null) {
                new SessionsLocalDataSource()
                        .markOfflineAsReadyForSync();
                IBGSessionServiceLocator.getBatchEvaluator().evaluate(getNoneFilter());
            }
        } catch (OutOfMemoryError | Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Uncaught exception failed to execute due to an exception", e);
        }

        if (defaultUncaughtExceptionHandler != null) {
            defaultUncaughtExceptionHandler.uncaughtException(thread, throwable);
        }
    }

    private String getStackTrace(final Throwable throwable) {
        final StringWriter sw = new StringWriter();
        final PrintWriter pw = new PrintWriter(sw, true);
        throwable.printStackTrace(pw);
        return sw.getBuffer().toString();
    }

    private boolean isReactNativeCrash(String stackTrace) {
        return stackTrace.contains("com.facebook.react.modules");
    }
}
