package com.instabug.library

import com.instabug.library.networkv2.execptions.RequestInterruptedException
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider

abstract class InstabugNetworkJob {

    protected fun interface JobErrorCallback {
        fun onError(exception: Exception)
    }

    protected fun enqueueJob(identifier: String, runnable: Runnable) {
        PoolProvider.getNetworkingSingleThreadExecutor(identifier).execute {
            executeRunnable(identifier, runnable)
        }
    }

    /**
     * Enqueue a network job that is could be retried multiple times if an error (network error)
     * is occurred while performing the job request.
     *
     * @param identifier The identifier to be used to get a unique executor to perform the job.
     * @param runnable that need to be executed.
     * @param errorCallback A callback that will be used to notify the caller in case of any error
     *        is occurred, if the error type was an InterruptedException we will ignore it as this
     *        type of error is thrown by the {@link com.instabug.library.networkv2.NetworkManager}
     *        if the current running job is canceled.
     *        You can ignore passing this callback and catch the exception in the caller function
     *        if you want to handle the error on your own.
     */
    @JvmOverloads
    protected fun enqueueRetryingJob(
        identifier: String,
        runnable: Runnable,
        errorCallback: JobErrorCallback? = null
    ) {
        PoolProvider.getNetworkingSingleThreadExecutor(
            identifier, true
        ).execute {
            try {
                executeRunnable(identifier, runnable)
            } catch (exception: Exception) {
                if (exception !is RequestInterruptedException)
                    errorCallback?.onError(exception)
            }
        }
    }

    private fun executeRunnable(identifier: String, runnable: Runnable) {
        if (Instabug.getApplicationContext() != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "$identifier Started")
            runnable.run()
        } else {
            InstabugSDKLogger.d(
                Constants.LOG_TAG, "Context was null while trying to start job: $identifier"
            )
        }
    }

    abstract fun start()

}