package com.instabug.library;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.LayoutRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.fragment.app.Fragment;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;

/**
 * Fragment class to unify implementation methodologies across all Instabug Fragment <br/>
 * Best practice would be: <br/>
 * * Don't override empty arguments constructor to send data to fragment
 * <br/>
 * * Create a {@code public static Fragment newInstance(...)} which returns an instance of this fragment.
 * <br/>
 * Example:
 * <pre>{@code
 * public static Fragment newInstance(String valueToInitWith){
 *      MyFragment fragment = new MyFragment();
 *      Bundle args = new Bundle();
 *      args.putString("value_to_init_with", valueToInitWith);
 *      fragment.setArguments(args);
 *      return fragment;
 * }}</pre>
 * * Instantiate your fragment using {@code MyFragment.newInstance("helloWorld");}
 * <br/>
 * * This creates an instance of {@code MyFragment} where {@link #getArguments()} contains a String whose key is {@code value_to_init_with} and value is {@code helloWorld}
 * <br/>
 * * Implement {@link #consumeNewInstanceSavedArguments()} to fetch data from {@code getArguments()}
 * <br/>
 * Example:
 * <pre>{@code
 * protected void consumeNewInstanceSavedArguments() {
 *      myValueVariable = getArguments().getString("value_to_init_with");
 * }}</pre>
 * * Implement {@link #getLayout()} to return your xml file for this fragment
 * <br/>
 * Example:
 * <pre>{@code
 * protected int getLayout() {
 *      return R.layout.instabug_lyt_feedback;
 * }}</pre>
 * * <b>if your fragment should have a title</b> Make sure your fragment layout contains {@code TextView} with {@code android:id="@+id/instabug_fragment_title"}
 * * Implement {@link #getTitle()} to return your {@code @StringRes} string id (to set dynamic fragment title check {@link #setTitle(String)})
 * <br/>
 * Example:
 * <pre>{@code
 * protected int getTitle() {
 *      return R.string.hello_world;
 * }}</pre>
 * * Implement {@link #saveState(Bundle)}, which will be called when {@link #onSaveInstanceState(Bundle)} is called <br/>
 * * Implement {@link #restoreState(Bundle)} which will be called when {@link #onViewCreated(View, Bundle)} is called iff there is state saved <br/>
 * * Use {@link #isStateRestored()} to make sure you don't initialize data that is already restored <br/>
 * Note: To understand more about save/restore states read <a href="http://inthecheesefactory.com/blog/fragment-state-saving-best-practices/en">this blog post</a>
 *
 * @author mSobhy
 */
public abstract class InstabugBaseFragment extends Fragment {

    @Nullable
    private View view;
    @Nullable
    private Activity activity;
    private boolean isStateRestored;

    /**
     * @return Layout res id to inflate for this fragment
     */
    protected abstract
    @LayoutRes
    int getLayout();

    /**
     * @return String of this fragment title (to ensure localization)
     */
    protected abstract String getTitle();

    /**
     * Allows for consuming/fetching any variables saved while initializing fragment <i>if any</i>
     *
     * @see <a href="http://stackoverflow.com/a/9245510/1328476">This SO answer</a>
     * @see #getArguments()
     */
    protected abstract void consumeNewInstanceSavedArguments();

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);
        this.activity = activity;
    }

    /**
     * * Calls {@code super.onCreate(Bundle)}
     * * Calls {@link #consumeNewInstanceSavedArguments()} <b>iff {@link #getArguments()} != null</b> <br/>
     *
     * @param savedInstanceState {@inheritDoc}
     */
    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getArguments() != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Arguments found, calling consumeNewInstanceSavedArguments with " + getArguments());
            consumeNewInstanceSavedArguments();
        }
        isStateRestored = false;
    }

    /**
     * Takes care of inflating the view and setting fragment title.<br/>
     * Sequence is as follows: <br/>
     * * Inflates {@link #getLayout()} <br/>
     * * Calls {@link #setTitle(String)}
     *
     * @param inflater           {@inheritDoc}
     * @param container          {@inheritDoc}
     * @param savedInstanceState {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        view = inflater.inflate(getLayout(), container, false);
        setTitle(getTitle());
        return view;
    }

    /**
     * Restore state of this fragment variable from {@code savedInstanceState} saved by {@code saveState(Bundle)}
     *
     * @param savedInstanceState bundle containing all saved values from {@code saveState(Bundle)}
     * @see #saveState(Bundle)
     */
    protected abstract void restoreState(Bundle savedInstanceState);

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        if (savedInstanceState != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "savedInstanceState found, calling restoreState");
            restoreState(savedInstanceState);
            isStateRestored = true;
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
        super.onSaveInstanceState(outState);
        InstabugSDKLogger.v(Constants.LOG_TAG, "onSaveInstanceState called, calling saveState");
        saveState(outState);
    }

    @Override
    public void onPause() {
        super.onPause();
    }

    @Override
    public void onResume() {
        super.onResume();
    }

    /**
     * Saves state of this fragment variable to {@code outState} to be restored using {@code restoreState(Bundle)}
     *
     * @param outState bundle to save variables, that will restored, to
     * @see #restoreState(Bundle)
     */
    protected abstract void saveState(Bundle outState);

    /**
     * State of this fragment restoration
     *
     * @return {@code true} if {@code savedInstanceState != null}, {@code false} otherwise (meaning there has been no old state for this fragment)
     * @see #restoreState(Bundle)
     */
    public boolean isStateRestored() {
        return isStateRestored;
    }

    /**
     * @return the instance of activity preserved when {@link #onAttach(Activity)} is called
     */
    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public Activity getPreservedActivity() {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Returning preserved activity " + activity);
        return activity;
    }

    /**
     * Checks if {@code R.id.instabug_fragment_title} TextView exists, if it does set it to {@code title}
     *
     * @param title to be set to fragment
     */
    public void setTitle(String title) {
        if (view == null) {
            return;
        }
        TextView titleTextView = (TextView) view.findViewById(R.id.instabug_fragment_title);
        if (titleTextView != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Setting fragment title to \"" + title + "\"");
            titleTextView.setText(title);
        }
    }

    protected String getLocalizedString(@StringRes int resourceId){
        return LocaleUtils.getLocaleStringResource(
                InstabugCore.getLocale(requireContext()),
                resourceId,
                requireContext());
    }
}
