package com.instabug.library.visualusersteps

import android.annotation.SuppressLint
import android.graphics.drawable.Drawable
import android.net.Uri
import android.text.TextUtils
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.CompoundButton
import android.widget.ImageButton
import android.widget.ImageView
import android.widget.SeekBar
import android.widget.TextView
import androidx.annotation.VisibleForTesting
import androidx.appcompat.view.menu.MenuItemImpl
import com.google.android.material.bottomnavigation.BottomNavigationItemView
import com.google.android.material.internal.NavigationMenuItemView
import com.google.android.material.tabs.TabLayout
import com.instabug.library.Constants
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.servicelocator.CoreServiceLocator.reproStepsProxy
import com.instabug.library.internal.servicelocator.CoreServiceLocator.touchedViewExtractorExtension
import com.instabug.library.util.BitmapUtils
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.StringUtility
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.SimpleCompletableFuture
import java.lang.ref.WeakReference
import java.util.concurrent.Future
import java.util.concurrent.FutureTask


private const val SEPARATOR = " - "
private const val UI_PRE_STRING = "UI that contains \"%s\""
private const val THE_BUTTON_PRE_STRING = "the button \"%s\""
private const val A_BUTTON_PRE_STRING = "a button"
private const val THE_BUTTON_ICON_PRE_STRING = "the button "
private const val THE_LABEL_PRE_STRING = "the label \"%s\""
private const val A_LABEL_PRE_STRING = "a label"
private const val THE_IMAGE_PRE_STRING = "the image \"%s\""
private const val AN_IMAGE_PRE_STRING = "an image"
private const val THE_SEEKBAR_PRE_STRING = "the slider \"%s\" to %d"
private const val A_SEEKBAR_PRE_STRING = "a slider to %d"
private const val THE_SWITCH_PRE_STRING = "the switch \"%s\""
private const val A_SWITCH_PRE_STRING = "a switch"
private const val MAX_SCANNED_VIEWS = 60
private const val MAX_CAPTURED_LABELS = 20
private const val MAX_LABEL_LENGTH = 500

/**
 * A class that is responsible for processing a touch on the UI elements to catch the most prominent labels
 * of that UI element to help in producing more descriptive [VisualUserSteps]
 *
 *
 * A most prominent label of a [View] can be a text or a content description (Accessibility Labels) of [TextView] of
 * [Button], or [ImageView]
 *
 *
 * A most prominent label of a [ViewGroup] contains children views is the collection of the text of
 * the first 20 [TextView] inside this view group while scanning first 60 views in the first 3 hierarchy levels
 *e.g
 *
 * "the button \"Click me\""
 * "the button icon"
 * "the label \"Information\""
 * "the image \"A beautiful image\""
 * "the slider \"Volume\" to 50"
 * "the switch \"Enable Feature\""
 * "UI that contains \"the button icon\""
 * "the button \"Home\""
 *
 */
class TouchedViewsProcessor private constructor() {
    fun composeProminentLabelForButtonWith(text: String?, withIcon: Boolean): String =
        if (withIcon)
            THE_BUTTON_ICON_PRE_STRING
        else
            text?.takeUnless { it.isBlank() }
                ?.let { THE_BUTTON_PRE_STRING.format(it) }
                ?: A_BUTTON_PRE_STRING

    /**
     * Get the most prominent label from the given [View].
     * @param view The touched view.
     * @return A [Future] containing the [TouchedView] with the most prominent label.
     */
    fun getMostProminentLabelFrom(view: View): Future<TouchedView?>? {
        var touchedView = TouchedView()
        touchedView.parent = reproStepsProxy.getCurrentParent()
        return touchedViewExtractorExtension
            ?.extract(view, touchedView)
            ?.let { reactNativeTouchedView -> touchedView = reactNativeTouchedView }
            ?.let { touchedView.touchedViewFutureTask() }
            ?: takeUnless { touchedViewExtractorExtension?.shouldDependOnNative == false }
                ?.let { touchedView.extractNativeTouchedView(view) }
    }

    private fun TouchedView.extractNativeTouchedView(view: View): Future<TouchedView?> {
        return when {
            ViewsTypeDetector.isButtonView(view) -> handleTouchedButton(view as Button)

            ViewsTypeDetector.isTextView(view) -> handleTouchedTextView(view as TextView)

            ViewsTypeDetector.TabLayout(view) -> handleTouchedTabView(view as TabLayout)

            ViewsTypeDetector.isNavItemView(view) -> handleTouchedNavView(view)

            ViewsTypeDetector.isImageButton(view) -> handleTouchedImageButton(view as ImageButton)

            ViewsTypeDetector.isImageView(view) -> handleTouchedImageView(view as ImageView)

            ViewsTypeDetector.isSwitch(view) -> handleTouchedSwitch(view as CompoundButton)

            ViewsTypeDetector.isSeekbar(view) -> handleSeekbar(view as SeekBar)

            view is ViewGroup -> detectLabelsOfUI(view)

            else -> touchedViewFutureTask()
        }
    }

    private fun TouchedView.handleSeekbar(view: SeekBar): Future<TouchedView?> {

        prominentLabel = when {
            view.hasAccessibilityLabel() -> THE_SEEKBAR_PRE_STRING
                .format(view.contentDescription, view.progress)

            else -> A_SEEKBAR_PRE_STRING.format(view.progress)
        }
        return touchedViewFutureTask()
    }

    private fun TouchedView.handleTouchedSwitch(view: CompoundButton): Future<TouchedView?> {
        prominentLabel = when {
            view.hasText() -> THE_SWITCH_PRE_STRING.format(view.text)
            view.hasAccessibilityLabel() -> THE_SWITCH_PRE_STRING.format(view.contentDescription)
            else -> A_SWITCH_PRE_STRING
        }
        return touchedViewFutureTask()
    }


    private fun TouchedView.detectLabelsOfUI(baseView: ViewGroup): Future<TouchedView?> {

        val viewGroupProcessor = GroupViewTouchedViewProcessor(baseView)

        //Get the collected labels
        prominentLabel = viewGroupProcessor.getCapturedLabel()

        return touchedViewFutureTask()
    }


    /**
     * Get the accessibility label `getContentDescription` of the [ImageView] if Available
     * Otherwise, return a pre defined string for anonymous [ImageView]
     */
    private fun TouchedView.handleTouchedImageView(view: ImageView): Future<TouchedView?> {
        prominentLabel = when {
            view.isPrivate() -> AN_IMAGE_PRE_STRING
            view.hasAccessibilityLabel() -> THE_IMAGE_PRE_STRING.format(view.contentDescription)
            else -> AN_IMAGE_PRE_STRING
        }
        return touchedViewFutureTask()
    }

    /**
     * Get the label of the [TextView], Text first, if not available then accessibility label `getContentDescription`
     * Otherwise, return a pre defined string for anonymous [TextView]
     */
    private fun TouchedView.handleTouchedTextView(view: TextView): Future<TouchedView?> {
        prominentLabel = when {
            view.isPrivate() -> A_LABEL_PRE_STRING
            view.hasText() -> view.text.toString().trimLabel()
                .let { THE_LABEL_PRE_STRING.format(it) }

            view.hasAccessibilityLabel() -> view.contentDescription.toString().trimLabel()
                .let { THE_BUTTON_PRE_STRING.format(it) }

            else -> A_LABEL_PRE_STRING
        }

        return touchedViewFutureTask()
    }


    /**
     * Get the label of the [Button], Text first, if not available then accessibility label `getContentDescription`
     * Otherwise capture the button icon if available, or return a pre defined string for anonymous [Button]
     */
    private fun TouchedView.handleTouchedButton(view: Button): Future<TouchedView?> {
        prominentLabel = when {
            view.isPrivate() -> A_BUTTON_PRE_STRING
            view.hasText() -> THE_BUTTON_PRE_STRING.format(view.text)
            else -> {
                val iconButtonFuture = getButtonIcon(view)?.let { captureButtonIcon(view, it) }

                if (iconButtonFuture != null) return iconButtonFuture
                else buttonLabelFromAccessibility(view)
            }
        }
        return touchedViewFutureTask()
    }


    private fun buttonLabelFromAccessibility(view: Button) = when {
        view.hasAccessibilityLabel() -> THE_BUTTON_PRE_STRING.format(view.contentDescription)
        else -> A_BUTTON_PRE_STRING

    }

    /**
     * Get the accessibility label `getContentDescription` if available of the [ImageButton],
     * Otherwise capture the button icon if available, or return a pre defined string for anonymous [Button]
     */
    private fun TouchedView.handleTouchedImageButton(view: ImageButton): Future<TouchedView?> {
        prominentLabel = when {
            view.isPrivate() -> A_BUTTON_PRE_STRING
            view.hasAccessibilityLabel() -> THE_BUTTON_PRE_STRING.format(view.contentDescription)
            view.drawable != null -> return captureButtonIcon(view, view.drawable)
            else -> A_BUTTON_PRE_STRING
        }
        return touchedViewFutureTask()
    }


    /**
     * Similar to {[.handleTouchedButton]
     * Get the label of the [com.google.android.material.tabs.TabLayout], Text first.
     * Otherwise capture the tab icon if available, or return a pre defined string for anonymous [Button]
     */
    private fun TouchedView.handleTouchedTabView(view: TabLayout): Future<TouchedView?> {
        val touchedViewCompletableFuture = SimpleCompletableFuture<TouchedView?>()
        view.addOnTabSelectedListener(
            createTabSelectedListener(this, view, touchedViewCompletableFuture)
        )
        return touchedViewCompletableFuture
    }

    private fun createTabSelectedListener(
        touchedView: TouchedView,
        view: TabLayout,
        touchedViewCompletableFuture: SimpleCompletableFuture<TouchedView?>
    ) = object : TabLayout.BaseOnTabSelectedListener<TabLayout.Tab?> {
        override fun onTabSelected(tab: TabLayout.Tab?) {
            handleSelectedTab(tab, view)
        }

        override fun onTabReselected(tab: TabLayout.Tab?) {
            handleSelectedTab(tab, view)
        }

        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        private fun handleSelectedTab(tab: TabLayout.Tab?, view: TabLayout) {
            tab?.let { touchedView.handleTabSelected(tab, touchedViewCompletableFuture, view) }
                ?: touchedViewCompletableFuture.complete(touchedView)
            view.removeOnTabSelectedListener(this)
        }

        override fun onTabUnselected(p0: TabLayout.Tab?) {
            //No-Operations
        }
    }

    private fun TouchedView.handleTabSelected(
        tab: TabLayout.Tab,
        touchedViewCompletableFuture: SimpleCompletableFuture<TouchedView?>,
        view: TabLayout
    ) {
        prominentLabel = when {
            !TextUtils.isEmpty(tab.text) -> THE_BUTTON_PRE_STRING.format(tab.text)
            tab.icon != null && !view.isPrivate() -> {
                touchedViewCompletableFuture.complete(captureTabIcon(tab.icon))
                return
            }

            !TextUtils.isEmpty(tab.contentDescription) -> THE_BUTTON_PRE_STRING.format(tab.contentDescription)

            else -> A_BUTTON_PRE_STRING
        }
        touchedViewCompletableFuture.complete(this)
    }

    /**
     * Similar to {[.handleTouchedButton]
     * Get the label of the [NavigationMenuItemView] or [BottomNavigationItemView], Text first.
     * Otherwise capture the tab icon if available, or return a pre defined string for anonymous [Button]
     */
    @SuppressLint("RestrictedApi")
    private fun TouchedView.handleTouchedNavView(view: View): Future<TouchedView?> {
        val itemData: MenuItemImpl? = getItemData(view)
        prominentLabel = itemData?.let {
            when {
                !TextUtils.isEmpty(itemData.title) -> THE_BUTTON_PRE_STRING.format(itemData.title)

                itemData.icon != null && !view.isPrivate() -> return captureButtonIcon(
                    view, itemData.icon
                )

                !TextUtils.isEmpty(itemData.contentDescription) -> THE_BUTTON_PRE_STRING.format(
                    itemData.contentDescription
                )

                else -> A_BUTTON_PRE_STRING
            }
        }
        return touchedViewFutureTask()
    }


    @SuppressLint("RestrictedApi")
    private fun getItemData(view: View) = when (view) {
        is NavigationMenuItemView -> view.itemData
        is BottomNavigationItemView -> view.itemData
        else -> null
    }

    @VisibleForTesting
    fun getImageButtonIcon(view: ImageButton): Drawable {
        return view.drawable
    }

    private fun TouchedView.captureButtonIcon(
        view: View, drawable: Drawable?
    ): Future<TouchedView?> {
        val currentTime = System.currentTimeMillis()
        return PoolProvider.submitIOTask {
            try {
                val absolutePath = BitmapUtils.saveDrawableBitmap(drawable, currentTime)
                if (absolutePath != null)
                    onDrawableSaved(absolutePath)
                else
                    handleDrawableFailed()
            } catch (throwable: Throwable) {
                onDrawableFailed(view, throwable)
            }
            this
        }
    }

    private fun TouchedView.onDrawableFailed(view: View, throwable: Throwable) {
        if (throwable.message != null) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "Error saving button icon bitmap: " + throwable.message
            )
        }
        when {
            view.hasAccessibilityLabel() -> prominentLabel =
                THE_BUTTON_PRE_STRING.format(view.contentDescription)

            else -> handleDrawableFailed()

        }
    }

    private fun TouchedView.handleDrawableFailed() {
        prominentLabel = A_BUTTON_PRE_STRING
        iconURI = null
        iconName = null
    }

    private fun TouchedView.onDrawableSaved(absolutePath: Uri) {
        prominentLabel = THE_BUTTON_ICON_PRE_STRING
        iconURI = absolutePath.toString()
        iconName = absolutePath.lastPathSegment
        if (absolutePath.path != null) {
            InstabugCore.encryptBeforeMarshmallow(absolutePath.path!!)
        }
    }

    /**
     * @return the directory where the icon to be saved.
     */
    private fun TouchedView.captureTabIcon(drawable: Drawable?): Future<TouchedView?> {
        val currentTime = System.currentTimeMillis()
        return PoolProvider.submitIOTask {
            try {
                val absolutePath = BitmapUtils.saveDrawableBitmap(drawable, currentTime)
                if (absolutePath != null)
                    onDrawableSaved(absolutePath)
                else
                    handleDrawableFailed()
            } catch (throwable: Throwable) {
                if (throwable.message != null) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG, "Error while saving tab icon: " + throwable.message
                    )
                }
                handleDrawableFailed()
            }
            this
        }
    }

    /**
     * @param button the touched button view
     * @return the drawable attached to the button
     */
    @VisibleForTesting
    fun getButtonIcon(button: Button): Drawable? =
        button.compoundDrawables.filterNotNull().firstOrNull()


    companion object {


        @JvmStatic
        val instance: TouchedViewsProcessor by lazy {
            TouchedViewsProcessor()
        }
    }


}

class GroupViewTouchedViewProcessor(
    baseView: ViewGroup,
    private val labelTemplate: String = UI_PRE_STRING
) {
    @VisibleForTesting
    var trimmedLabels: MutableList<LabelFrame> = ArrayList()
        private set

    @VisibleForTesting
    var flattenHierarchy: MutableList<WeakReference<View>> = ArrayList()
        private set
    private var firstLevelViews: MutableList<WeakReference<ViewGroup>> = ArrayList()
    private var secondLevelViews: MutableList<WeakReference<ViewGroup>> = ArrayList()
    private var thirdLevelViews: MutableList<WeakReference<ViewGroup>> = ArrayList()
    private var capturedLabel: StringBuilder? = StringBuilder()

    init {
        //Flatten the hierarchy
        flatHierarchy(baseView)

        //Trim Captured Labels
        trimCapturedLabels()

        //Sort Trimmed Labels
        sortTrimmedLabels()
    }

    /**
     * Build up and return the output label of the touched [ViewGroup]
     */
    fun getCapturedLabel(): String? {
        val label = capturedLabel ?: return null
        for (trimmedLabel in trimmedLabels) {
            var textToBeAppended = trimmedLabel.text
            val shouldAppendSeparator = label.isNotEmpty()
            var remainingLength = MAX_LABEL_LENGTH - label.length
            if (shouldAppendSeparator) {
                remainingLength -= SEPARATOR.length
            }
            if (remainingLength <= 0) {
                break
            }
            textToBeAppended = StringUtility.trimString(textToBeAppended, remainingLength)
            if (shouldAppendSeparator) {
                label.append(SEPARATOR)
            }
            label.append(textToBeAppended)
        }
        return if (label.isNotBlank()) labelTemplate.format(label)
        else null
    }

    /**
     * Breadth First Traversal (BFS) technique to convert the hierarchy of [ViewGroup] into a flat list of views
     * Targeting the original level of the [ViewGroup] children and 3 more deeper levels inside these children
     * Î
     */
    private fun flatHierarchy(baseView: ViewGroup) {
        // Collect Original Deeper Level Views
        addViewToFlatHierarchy(baseView, firstLevelViews)
        // Collect First Deeper Level Views
        scanHierarchyLevelViews(firstLevelViews, secondLevelViews)
        // Collect Second Deeper Level Views
        scanHierarchyLevelViews(secondLevelViews, thirdLevelViews)
        // Collect Third Deeper Level  Views
        scanHierarchyLevelViews(thirdLevelViews, null)
    }

    private fun scanHierarchyLevelViews(
        currentLevelViews: List<WeakReference<ViewGroup>>,
        deeperLevelViews: MutableList<WeakReference<ViewGroup>>?
    ) {
        for (i in currentLevelViews.indices) {
            if (flattenHierarchy.size >= MAX_SCANNED_VIEWS) {
                break
            }

            currentLevelViews[i].get()?.let { addViewToFlatHierarchy(it, deeperLevelViews) }
        }
    }

    private fun addViewToFlatHierarchy(
        currentView: ViewGroup, deeperLevelViews: MutableList<WeakReference<ViewGroup>>?
    ) {
        for (i in 0 until currentView.childCount) {
            if (flattenHierarchy.size >= MAX_SCANNED_VIEWS) break

            flattenHierarchy.add(WeakReference(currentView.getChildAt(i)))

            if (currentView.getChildAt(i) is ViewGroup && deeperLevelViews != null) {
                deeperLevelViews.add(WeakReference(currentView.getChildAt(i) as ViewGroup))
            }
        }
    }


    /**
     * Sort the collected [TextView] in `trimmedLabels` by nearest to the top-left of the window
     */
    private fun sortTrimmedLabels() {
        trimmedLabels.sortWith { o1, o2 -> o1.compareTo(o2) }
    }


    /**
     * Collect the first 20 view of type [TextView] of the flatten views list `flattenHierarchy`
     * and map them to list of [LabelFrame] holding the text value and (Y,X) of top-left point of the view
     */
    private fun trimCapturedLabels() {
        for (viewRef in flattenHierarchy) {
            val view = viewRef.get()
            if (view is TextView) {
                val viewLabel = getViewLabel(view as TextView?)
                if (!viewLabel.isNullOrEmpty()) {
                    val location = IntArray(2)
                    view.getLocationOnScreen(location)
                    val x = location[0]
                    val y = location[1]
                    trimmedLabels.add(LabelFrame(viewLabel, y.toFloat(), x.toFloat()))
                }
            }
            if (trimmedLabels.size == MAX_CAPTURED_LABELS) {
                break
            }
        }
    }

    private fun getViewLabel(textView: TextView?): String? {
        return if (textView.hasText() && textView?.isPrivate() == false) textView.text.toString()
        else {
            null
        }
    }
}

private fun TouchedView.touchedViewFutureTask() = FutureTask { this }.apply { run() }

private fun TextView?.hasText() = !this?.text.isNullOrBlank()

private fun View.hasAccessibilityLabel(): Boolean = !contentDescription.isNullOrBlank()
private fun View.isPrivate() = VisualUserStepsHelper.isPrivateView(this)
private fun String.trimLabel(): String = StringUtility.trimString(
    this, MAX_LABEL_LENGTH
)
