package com.instabug.library.visualusersteps

import android.content.Context
import androidx.annotation.WorkerThread
import com.instabug.library.SpanIDProvider
import com.instabug.library.WatchableSpansCacheDirectory
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.ifNotExists
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService
import java.io.File

private const val EXEC_QUEUE_ID = "repro-screenshots-dir-op-exec"
private const val MAX_ALLOWED_DIRS = 5
private const val MAX_ALLOWED_DIRS_DELTA = 1

class ReproScreenshotsCacheDirectory(
    private val executor: OrderedExecutorService,
    private val ctxGetter: () -> Context?,
    private val baseDirectoryGetter: (Context) -> File?,
    spanIDProvider: SpanIDProvider
) : WatchableSpansCacheDirectory {
    private val currentALID: String = spanIDProvider.spanId
    private val watchersMap: MutableMap<Int, Boolean> = mutableMapOf()

    init {
        executor.execute(EXEC_QUEUE_ID) { trimIfNeeded(getOldDirs(false)) }
    }

    private val internalFilesDirectory: File?
        @WorkerThread get() = ctxGetter()?.let(baseDirectoryGetter)
            ?.let(Helper::getReproScreenshotsDirectory)
    override val filesDirectory: File?
        @WorkerThread get() = executor.submit(EXEC_QUEUE_ID) {
            internalFilesDirectory?.ifNotExists { mkdirs() }
        }.get()
    override val currentSpanDirectory: File?
        @WorkerThread get() = executor.submit(EXEC_QUEUE_ID) {
            internalFilesDirectory
                ?.let { Helper.getALIDDirectory(it, currentALID) }
                ?.ifNotExists { mkdirs() }
        }.get()
    override val oldSpansDirectories: List<File>
        @WorkerThread get() = executor.submit(EXEC_QUEUE_ID) { getOldDirs(true) }.get()

    override fun setCurrentSpanId(spanId: String?) {
        // No-Op
    }

    override fun addWatcher(watcherId: Int) = executor.execute(EXEC_QUEUE_ID) {
        if (!watchersMap.containsKey(watcherId)) {
            watchersMap[watcherId] = false
        }
    }

    override fun consentOnCleansing(watcherId: Int) = executor.execute(EXEC_QUEUE_ID) {
        if (watchersMap.containsKey(watcherId)) {
            watchersMap[watcherId] = true
            cleanseIfNeeded()
        }
    }

    override fun removeWatcher(watcherId: Int) = executor.execute(EXEC_QUEUE_ID) {
        if (watchersMap.containsKey(watcherId)) {
            watchersMap.remove(watcherId)
            cleanseIfNeeded()
        }
    }

    @WorkerThread
    override fun queryWatcherConsent(watcherId: Int): Boolean? =
        executor.submit(EXEC_QUEUE_ID) { watchersMap[watcherId] }.get()

    private fun cleanseIfNeeded() = runCatching {
        if (watchersMap.all { (_, hasConsented) -> hasConsented }) {
            getOldDirs(false).forEach { dir -> dir.deleteRecursively() }
            watchersMap.keys.forEach { watcher -> watchersMap[watcher] = false }
        }
    }.runOrLogAndReport("Couldn't cleanse repro screenshots dirs.")

    private fun getOldDirs(includeCurrent: Boolean): List<File> = runCatching {
        internalFilesDirectory?.takeIf { it.exists() }
            ?.listFiles { file -> file.name != currentALID || includeCurrent }
            ?.toList() ?: emptyList()
    }.getOrLogAndReport(emptyList(), "Couldn't retrieve repro screenshots old dirs.")

    private fun trimIfNeeded(oldDirs: List<File>) = runCatching {
        if (oldDirs.size >= MAX_ALLOWED_DIRS) {
            val delta = oldDirs.size - MAX_ALLOWED_DIRS + MAX_ALLOWED_DIRS_DELTA
            reportNonfatalIfMaxDeltaExceeded(delta)
            val operableDirsList = oldDirs
                .sortedByDescending { dir -> dir.lastModified() }
                .toMutableList()
            repeat(delta) { operableDirsList.removeLastOrNull()?.deleteRecursively() }
        }
    }.runOrLogAndReport("Couldn't trim repro screenshots old dirs.")

    private fun reportNonfatalIfMaxDeltaExceeded(dirsDelta: Int) = runCatching {
        if (dirsDelta > MAX_ALLOWED_DIRS_DELTA) error("Max delta exceeded.")
    }.runOrLogAndReport("Repro screenshots dirs exceeded max allowed delta.")

    object Helper {
        private const val REPRO_SCREENSHOTS_DIR_NAME = "repro-screenshots"

        fun getReproScreenshotsDirectory(baseDirectory: File): File =
            File(baseDirectory, REPRO_SCREENSHOTS_DIR_NAME)

        fun getALIDDirectory(screenshotsDirectory: File, alid: String): File =
            File(screenshotsDirectory, alid)
    }
}