package com.instabug.library.visualusersteps

import android.graphics.Bitmap
import android.view.View
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.IBGFeature
import com.instabug.library.SpansCacheDirectory
import com.instabug.library.core.InstabugCore
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.screenshot.ScreenshotCaptor
import com.instabug.library.screenshot.instacapture.ScreenshotRequest
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.util.threading.SingleThreadPoolExecutor
import com.instabug.library.visualusersteps.manual.LOG_FEATURE_DISABLED_DUE_TO_REPRO_STEPS
import java.lang.ref.WeakReference
import java.util.concurrent.Future

class ReproCapturingNotAuthorizedException(message: String) : RuntimeException(message)

private fun notAuthorized(message: String) = ReproCapturingNotAuthorizedException(message)

/**
 * A proxy contract for repro features.
 * Should be implemented as part of delegation, forming a mutual auth proxy with interested
 * parties (features) authorization capabilities to perform its original job.
 */
interface ReproCapturingProxy {
    /**
     * Whether this proxy is authorized or not
     */
    val isAuthorized: Boolean

    /**
     * Evaluate a [ReproConfigurationsProvider].
     * [ReproConfigurationsProvider] is the contract representing repro features state for interested
     * parties (features). When evaluating, the proxy of a specific repro feature will be depending on
     * a defined flag in [ReproConfigurationsProvider], being enabled authorizes the proxy & disabled revokes
     * the previous evaluations auth.
     * @param configProvider the [ReproConfigurationsProvider] that should be used in evaluating the auth state
     * of a given interested party
     */
    fun evaluate(configProvider: ReproConfigurationsProvider)
}

abstract class AbstractReproCapturingProxy : ReproCapturingProxy {
    private val authorizations: MutableSet<Int> = mutableSetOf()

    protected open val internalIsAuthorized: Boolean
        get() = authorizations.isNotEmpty()

    override val isAuthorized: Boolean
        @WorkerThread get() = submitAndGet { internalIsAuthorized }

    /**
     * A functional evaluator, given an object of [ReproConfigurationsProvider] it should contain
     * the specifics of how a certain proxy depends on [ReproConfigurationsProvider] flags.
     */
    protected abstract val evaluator: (ReproConfigurationsProvider) -> Boolean

    override fun evaluate(configProvider: ReproConfigurationsProvider) = execute {
        with(configProvider) {
            if (evaluator(this)) authorize(reproProxyAuthId) else revoke(reproProxyAuthId)
        }
        postEvaluate()
    }

    protected open fun postEvaluate() {
        // No-Op
    }

    abstract fun execute(job: () -> Unit)

    abstract fun <V> submitAndGet(job: () -> V): V

    private fun authorize(authId: Int) {
        if (authorizations.contains(authId)) return
        authorizations.add(authId)
    }

    private fun revoke(authId: Int) {
        if (!authorizations.contains(authId)) return
        authorizations.remove(authId)
    }
}

interface ReproScreenshotsCapturingProxy : ReproCapturingProxy, ScreenshotCaptor

@VisibleForTesting
const val ERROR_REPRO_SCREENSHOTS_UNAUTHORIZED =
    "Repro screenshots capturing is disabled for all report types or feature not available"

private const val SCREENSHOT_CAPTURING_QUEUE_ID = "repro-screenshots-exec"

class BasicReproScreenshotsCapturingProxy(
    private val originalCaptor: ScreenshotCaptor,
    private val savingDirectory: SpansCacheDirectory,
    private val executor: OrderedExecutorService
) : AbstractReproCapturingProxy(), ReproScreenshotsCapturingProxy {

    override val evaluator: (ReproConfigurationsProvider) -> Boolean
        get() = { provider -> provider.isReproScreenshotsEnabled }

    override fun capture(request: ScreenshotRequest) {
        if (!internalIsAuthorized) {
            request.listener.onCapturingFailure(notAuthorized(ERROR_REPRO_SCREENSHOTS_UNAUTHORIZED))
            return
        }
        originalCaptor.capture(request)
    }

    override fun postEvaluate() {
        cleanseIfNeeded()
    }

    override fun execute(job: () -> Unit) = executor.execute(SCREENSHOT_CAPTURING_QUEUE_ID, job)

    override fun <V> submitAndGet(job: () -> V): V =
        executor.submit(SCREENSHOT_CAPTURING_QUEUE_ID, job).get()

    private fun cleanseIfNeeded() {
        if (internalIsAuthorized) return
        runCatching {
            savingDirectory.currentSpanDirectory
                ?.takeIf { dir -> dir.exists() }
                ?.deleteRecursively()
        }
    }
}

interface ReproStepsCapturingProxy : ReproCapturingProxy, ReproStepsCaptor

class BasicReproStepsCapturingProxy(
    private val originalCaptorCreator: () -> ReproStepsCaptor,
    private val executor: SingleThreadPoolExecutor
) : AbstractReproCapturingProxy(), ReproStepsCapturingProxy {
    override val internalIsAuthorized: Boolean
        get() =
            super.internalIsAuthorized && isEnabledByFeatureFlag

    private val isEnabledByFeatureFlag
        get() =
            InstabugCore.isFeatureAvailablePersistable(IBGFeature.REPRO_STEPS)

    @VisibleForTesting
    @Volatile
    var originalCaptor: ReproStepsCaptor? = null

    private fun ReproStepsCaptor?.createIfNeededOrReset(): ReproStepsCaptor? {
        return if (internalIsAuthorized) {
            getOrCreateOriginalCaptor()
        } else {
            this?.let {
                cleanAndReset()
                null
            }
        }
    }

    private fun ReproStepsCaptor?.getOrCreateOriginalCaptor(): ReproStepsCaptor {
        return this ?: originalCaptorCreator().also {
            originalCaptor = it
        }
    }

    override val evaluator: (ReproConfigurationsProvider) -> Boolean
        get() = { provider -> provider.isReproStepsEnabled }

    override fun postEvaluate() {
        cleanAndResetOrCreateIfNeeded()
    }

    override fun execute(job: () -> Unit) = executor.execute(job)

    override fun <V> submitAndGet(job: () -> V): V = executor.submit(job).get()

    override fun setLastView(viewRef: WeakReference<View>?) {
        execute {
            originalCaptor.createIfNeededOrReset()?.setLastView(viewRef)
        }
    }

    override fun getCurrentParent(): Parent? = kotlin.runCatching {
        originalCaptor?.getCurrentParent()
    }.getOrElse { throwable ->
        InstabugSDKLogger.e(LOG_TAG, "Error while calling getCurrentParent", throwable)
        null
    }

    override fun addVisualUserStep(screenName: String, bitmap: Bitmap?) {
        execute {
            originalCaptor.createIfNeededOrReset()?.addVisualUserStep(screenName, bitmap)
        }
    }

    override fun addVisualUserStep(
        stepType: String,
        screenName: String?,
        view: String?,
        icon: String?
    ) {
        execute {
            originalCaptor.createIfNeededOrReset()
                ?.addVisualUserStep(stepType, screenName, view, icon)
        }
    }


    override fun addVisualUserStep(
        stepType: String,
        screenName: String?,
        finalTarget: IBGUINode,
        touchedView: Future<TouchedView?>
    ) {
        execute {
            originalCaptor.createIfNeededOrReset()
                ?.addVisualUserStep(stepType, screenName, finalTarget, touchedView)
        }
    }

    override fun logKeyboardEvent(isKeyboardOpen: Boolean) {
        execute {
            originalCaptor.createIfNeededOrReset()?.logKeyboardEvent(isKeyboardOpen)
        }
    }

    override fun logFocusChange(oldFocus: View?, newFocus: View?) {
        execute {
            originalCaptor.createIfNeededOrReset()?.logFocusChange(oldFocus, newFocus)
        }
    }

    override fun logInstabugEnabledStep() {
        execute {
            originalCaptor.createIfNeededOrReset()?.logInstabugEnabledStep()
        }
    }

    override fun logForegroundStep() {
        execute {
            originalCaptor.createIfNeededOrReset()?.logForegroundStep()
        }
    }

    override fun logBackgroundStep() {
        execute {
            originalCaptor.createIfNeededOrReset()?.logBackgroundStep()
        }
    }

    override fun removeLastTapStep() {
        execute {
            originalCaptor.createIfNeededOrReset()?.removeLastTapStep()
        }
    }

    override fun removeScreenshotId(screenshotUri: String) {
        execute {
            originalCaptor.createIfNeededOrReset()?.removeScreenshotId(screenshotUri)
        }
    }

    override fun fetch(): ArrayList<VisualUserStep?> {
        return kotlin.runCatching { originalCaptor?.fetch() }.getOrElse { throwable ->
            InstabugSDKLogger.e(LOG_TAG, "Error while fetching VisualUserSteps", throwable)
            java.util.ArrayList()
        }
            ?: ArrayList()
    }

    override fun clean() {
        execute {
            originalCaptor.createIfNeededOrReset()?.clean()
        }
    }

    override fun reset() {
        execute {
            originalCaptor.createIfNeededOrReset()?.reset()
        }
    }

    override fun duplicateCurrentParent() {
        execute {
            originalCaptor.createIfNeededOrReset()?.duplicateCurrentParent()
                ?: InstabugSDKLogger.d(
                    LOG_TAG, LOG_FEATURE_DISABLED_DUE_TO_REPRO_STEPS
                )
        }
    }

    private fun cleanAndResetOrCreateIfNeeded() {
        if (internalIsAuthorized) {
            originalCaptor.getOrCreateOriginalCaptor()
            return
        }
        cleanAndReset()
    }

    private fun cleanAndReset() {
        if (originalCaptor == null) return
        originalCaptor?.run {
            clean()
            reset()
        }
        nullifyOriginalCaptor()
    }

    private fun nullifyOriginalCaptor() {
        originalCaptor = null
    }
}

class CompositeReproCapturingProxy(
    private val proxiesList: List<ReproCapturingProxy>
) : ReproCapturingProxy {
    override val isAuthorized: Boolean
        @WorkerThread get() = proxiesList.fold(true) { acc, proxy -> acc && proxy.isAuthorized }

    override fun evaluate(configProvider: ReproConfigurationsProvider) {
        proxiesList.forEach { proxy -> proxy.evaluate(configProvider) }
    }
}
