package com.instabug.library.util;

import androidx.annotation.IntRange;
import androidx.annotation.Keep;

import io.reactivexport.Observable;

@Keep
public class TaskDebouncer {

    private final long timeout;
    private long lastRun;

    public TaskDebouncer(@IntRange(from = 0) long timeoutMillis) {
        this.timeout = timeoutMillis;
    }

    /**
     * Executes the task only and only if the last execution time has timed out
     */
    public void debounce(Runnable runnable) {
        if (getElapsedTime() >= timeout) {
            updateLastRun(System.currentTimeMillis());
            runnable.run();
        }
    }

    public <T> Observable<T> debounce(Observable<T> observable) {
        if (getElapsedTime() >= timeout) {
            updateLastRun(System.currentTimeMillis());
            return observable;
        }
        return Observable.empty();
    }

    private void updateLastRun(long timeMillis) {
        this.lastRun = timeMillis;
    }

    private long getElapsedTime() {
        return System.currentTimeMillis() - lastRun;
    }

    public void resetLastRun() {
        lastRun = 0L;
    }
}
