package com.instabug.library.util

import android.graphics.Bitmap
import android.graphics.PorterDuff
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.annotation.ColorInt
import androidx.annotation.ColorRes
import androidx.core.content.ContextCompat
import com.instabug.library.*
import com.instabug.library.core.InstabugCore
import com.instabug.library.settings.SettingsManager

internal object PbiFooterUtils {

    @JvmStatic
    @JvmName("handlePbiFooter")
    internal fun handlePbiFooter(view: View?) {
        if (isWhiteLabelingEnabled() && !SettingsManager.getInstance().isCustomBrandingEnabled) {
            if (view != null) {
                hidePbiFooter(view)
            }
        } else if (view != null) {
            showPbiFooter(view)
        }
    }

    @JvmStatic
    private fun hidePbiFooter(view: View) {
        val container = view.findViewById<LinearLayout>(R.id.instabug_pbi_container) ?: return
        container.visibility = View.GONE
    }

    @JvmStatic
    private fun showPbiFooter(view: View) {
        val container = view.findViewById<LinearLayout>(R.id.instabug_pbi_container) ?: return
        val logoImageView = view.findViewById<ImageView>(R.id.image_instabug_logo) ?: return
        val poweredByTV = view.findViewById<TextView>(R.id.text_view_pb) ?: return
        val context = view.context ?: return

        container.visibility = View.VISIBLE

        if (!isWhiteLabelingEnabled()) {
            logoImageView.setImageResource(R.drawable.ibg_core_ic_instabug_logo)
            poweredByTV.text = LocaleUtils.getLocaleStringResource(
                InstabugCore.getLocale(context), R.string.instabug_str_powered_by_instabug, context
            )
        } else if (SettingsManager.getInstance().isCustomBrandingEnabled) {
            logoImageView.setImageBitmap(getCustomBrandingLogo())
            poweredByTV.visibility = View.GONE
        }
    }

    @JvmStatic
    @JvmName("handleThemeColor")
    internal fun handleThemeColor(
        view: View?,
        @ColorRes lightColor: Int? = R.color.pbi_footer_color_light,
        @ColorRes darkColor: Int? = R.color.pbi_footer_color_dark
    ) {
        if (isWhiteLabelingEnabled()) return
        val context = view?.context ?: return
        val instabugTheme = SettingsManager.getInstance().theme
        val targetColor: Int =
            if (instabugTheme == InstabugColorTheme.InstabugColorThemeLight && lightColor != null) {
                lightColor
            } else if (instabugTheme == InstabugColorTheme.InstabugColorThemeDark && darkColor != null) {
                darkColor
            } else {
                return
            }

        setThemeColor(view, ContextCompat.getColor(context, targetColor))
    }

    @JvmStatic
    @JvmName("setThemeColor")
    internal fun setThemeColor(view: View?, @ColorInt targetColor: Int) {
        if (isWhiteLabelingEnabled()) return

        val container = view?.findViewById<LinearLayout>(R.id.instabug_pbi_container) ?: return
        if (container.visibility == View.GONE || container.visibility == View.INVISIBLE) return

        val logoImageView = view.findViewById<ImageView>(R.id.image_instabug_logo) ?: return
        val poweredByTV = view.findViewById<TextView>(R.id.text_view_pb) ?: return

        logoImageView.setColorFilter(targetColor, PorterDuff.Mode.SRC_ATOP)
        poweredByTV.setTextColor(targetColor)
    }

    private fun getCustomBrandingLogo(): Bitmap? {
        return if (Instabug.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
            SettingsManager.getInstance().lightCustomBrandingLogo
        } else if (Instabug.getTheme() == InstabugColorTheme.InstabugColorThemeDark) {
            SettingsManager.getInstance().darkCustomBrandingLogo
        } else {
            null
        }
    }

    private fun isWhiteLabelingEnabled(): Boolean {
        return InstabugCore.getFeatureState(IBGFeature.WHITE_LABELING) == Feature.State.ENABLED
    }

}