package com.instabug.library.util;

import static com.instabug.library.internal.device.InstabugDeviceProperties.getPackageInfo;
import static com.instabug.library.util.overairversion.OverAirVersionType.EXPO;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.ActivityManager;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageInfo;
import android.content.res.Configuration;
import android.hardware.display.DisplayManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.os.ConfigurationCompat;
import androidx.core.os.LocaleListCompat;

import com.instabug.library.BuildConfig;
import com.instabug.library.BuildFieldsProvider;
import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.SessionManager;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.internal.device.InstabugRootChecker;
import com.instabug.library.util.overairversion.CodePushVersionHandler;
import com.instabug.library.util.overairversion.ExpoVersionHandler;
import com.instabug.library.util.overairversion.OverAirSharedConstants;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.ref.WeakReference;

/**
 * Created by tarek on 2/26/18.
 */

public class DeviceStateProvider {

    @Nullable
    private static WeakReference<File> externalStorageDirWeakRef = null;

    public static String getDevice() {
        return InstabugDeviceProperties.getDeviceType();
    }

    public static boolean isDeviceRooted() {
        try {
            if (InstabugRootChecker.isDeviceRooted())
                return true;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while checking if " +
                    "device is " +
                    "rooted or not " + e.getMessage());
            return false;
        }
        return false;
    }

    @NonNull
    public static String getOS() {
        return "OS Level " + Build.VERSION.SDK_INT;
    }

    public static String getCarrier(Context context) {
        try {
            return ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE))
                    .getNetworkOperatorName();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while get Carrier", e);
            return "Unknown";
        }
    }

    public static String getAppPackageName(Context context) {
        return InstabugDeviceProperties.getPackageName(context);
    }

    public static String getAppVersion(Context context) {
        String appVersion;
        PackageInfo packageInfo = getPackageInfo(context);
        appVersion = String.format("%s (%s)", packageInfo.versionName, packageInfo.versionCode);
        int storedType = OverAirSharedConstants.INSTANCE.getStoredType();
        if (storedType == EXPO) {
            return ExpoVersionHandler.INSTANCE.sanitize(appVersion);
        } else {
            return CodePushVersionHandler.INSTANCE.sanitize(appVersion);
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static int getBatteryLevel(Context context) {
        try {
            IntentFilter battery = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
            Intent currentBattery = context.registerReceiver(null, battery);
            if (currentBattery != null) {
                int level = currentBattery.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
                int scale = currentBattery.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
                float batteryPct = level / (float) scale;
                return (int) (batteryPct * 100);
            } else {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Could't obtain battery level");
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while get battery level", e);
        }
        return -1;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static String getBatteryState(Context context) {
        try {
            IntentFilter battery = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
            Intent batteryStatus = context.registerReceiver(null, battery);
            if (batteryStatus != null) {
                int status = batteryStatus.getIntExtra(BatteryManager.EXTRA_STATUS, -1);
                boolean isCharging = status == BatteryManager.BATTERY_STATUS_CHARGING ||
                        status == BatteryManager.BATTERY_STATUS_FULL;
                int chargePlug = batteryStatus.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1);
                boolean usbCharge = chargePlug == BatteryManager.BATTERY_PLUGGED_USB;
                boolean acCharge = chargePlug == BatteryManager.BATTERY_PLUGGED_AC;
                return (isCharging ? "Charging" + (acCharge ? " through AC Charger" :
                        (usbCharge ? " through USB cable" : "")) : "Unplugged");
            } else {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Could't obtain battery state");
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while get battery state", e);
        }
        return "Unknown";
    }

    public static boolean getWifiState(Context context) {
        try {
            ConnectivityManager connectivityManager = (ConnectivityManager) context
                    .getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo wifiInfo = connectivityManager.getNetworkInfo(ConnectivityManager
                    .TYPE_WIFI);
            return wifiInfo != null && wifiInfo.isConnected();
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while get wifi state", e);
            return false;
        }
    }


    public static long getFreeMemory(Context context) {
        ActivityManager activityManager = (ActivityManager) context.getSystemService(Context
                .ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
        activityManager.getMemoryInfo(memoryInfo);
        return (memoryInfo.availMem / (1024 * 1024));
    }

    public static long getUsedMemory(Context context) {
        ActivityManager activityManager = (ActivityManager) context.getSystemService(Context
                .ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
        activityManager.getMemoryInfo(memoryInfo);
        long usedMemory = (calculateTotalMemory(context) - memoryInfo.availMem);
        return (usedMemory / (1024 * 1024));
    }

    public static long getTotalMemory(Context context) {
        long totalMemory = calculateTotalMemory(context);
        if (totalMemory == 0) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculating total memory");
            return -1;
        } else {
            return (totalMemory / (1024 * 1024));
        }
    }

    private static long calculateTotalMemory(Context context) {
        if (BuildFieldsProvider.INSTANCE.provideBuildVersion() >= 16) {
            return calculateTotalMemoryApi16(context);
        } else {
            return calculateTotalMemoryPreApi16();
        }
    }

    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    private static long calculateTotalMemoryApi16(Context context) {
        ActivityManager actManager = (ActivityManager) context.getSystemService(Context
                .ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo memInfo = new ActivityManager.MemoryInfo();
        actManager.getMemoryInfo(memInfo);
        return memInfo.totalMem;
    }

    private static long calculateTotalMemoryPreApi16() {
        RandomAccessFile reader = null;
        try {
            //FIXME: StrictMode$StrictModeDiskReadViolation
            reader = new RandomAccessFile("/proc/meminfo", "r");
            String ram = reader.readLine();
            String[] splits = ram.split(":");
            ram = splits[1].trim();
            ram = ram.substring(0, ram.length() - 3).trim();
            reader.close();
            return Long.parseLong(ram) * 1024L;
        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            if (reader != null) try {
                reader.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return 0;
    }

    public static long getFreeInternalStorage() {
        Context context = getApplicationContext();
        if (context != null) {
            File internalDir = context.getFilesDir();
            if (internalDir != null) {
                long space = internalDir.getUsableSpace();
                return (space / (1024 * 1024));
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Got Error while calculating free storage");
                return -1;
            }
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Context was null while calculating free storage");
            return -1;
        }
    }


    public static long getFreeStorage() {
        try {
            File externalCacheDir = getExternalCacheDir();
            if (externalMemoryAvailable() && externalCacheDir != null) {
                long space = externalCacheDir.getUsableSpace();
                return (space / (1024 * 1024));
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate free storage");
                return -1;
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate free storage");
            return -1;
        }
    }

    public static long getUsedStorage() {
        try {
            File externalCacheDir = getExternalCacheDir();
            if (externalMemoryAvailable() && externalCacheDir != null) {
                long usedStorage = externalCacheDir.getTotalSpace() - externalCacheDir.getFreeSpace();
                return (usedStorage / (1024 * 1024));
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate used storage");
                return -1;
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate used storage");
            return -1;
        }
    }

    public static long getTotalStorage() {
        try {
            File externalCacheDir = getExternalCacheDir();
            if (externalMemoryAvailable() && externalCacheDir != null) {
                long totalStorage = externalCacheDir.getTotalSpace();
                return (int) (totalStorage / (1024 * 1024));
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate total storage");
                return -1;
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while calculate total storage");
            return -1;
        }
    }

    @Nullable
    private static File getExternalCacheDir() {
        Context context = getApplicationContext();
        if (context == null) {
            return null;
        } else if (externalStorageDirWeakRef == null || externalStorageDirWeakRef.get() == null) {
            File externalCacheDir = context.getExternalCacheDir();
            if (externalCacheDir != null) {
                externalStorageDirWeakRef = new WeakReference<>(externalCacheDir);
            }
        }
        return externalStorageDirWeakRef != null ? externalStorageDirWeakRef.get() : null;
    }

    @Nullable
    private static Context getApplicationContext() {
        return Instabug.getApplicationContext();
    }

    private static boolean externalMemoryAvailable() {
        try {
            return android.os.Environment.getExternalStorageState().equals(android.os.Environment
                    .MEDIA_MOUNTED);
        } catch (OutOfMemoryError | Exception exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't get external Memory Available", exception);
            return false;
        }
    }

    public static String getScreenDensity(Context context) {
        DisplayMetrics displayMetrics = getDisplayMetrics(context);
        if (displayMetrics.densityDpi < DisplayMetrics.DENSITY_MEDIUM) {
            return "ldpi";
        } else if (displayMetrics.densityDpi < DisplayMetrics.DENSITY_HIGH) {
            return "mdpi";
        } else if (displayMetrics.densityDpi < DisplayMetrics.DENSITY_XHIGH) {
            return "hdpi";
        } else if (displayMetrics.densityDpi < 480) {
            return "xhdpi";
        } else if (displayMetrics.densityDpi < 640) {
            return "xxhdpi";
        } else {
            return "xxxhdpi";
        }
    }

    public static String getScreenSize(Context context) {
        DisplayMetrics displayMetrics = getDisplayMetrics(context);
        return String.format("%sx%s", displayMetrics.widthPixels, displayMetrics.heightPixels);
    }

    public static DisplayMetrics getDisplayMetrics(Context context) {
        Display defaultDisplay;
        DisplayMetrics displayMetrics = new DisplayMetrics();

        /* Use DeviceManager to avoid android.os.context.getSystemService(Context.WINDOW_SERVICE.IncorrectContextUseViolation when StrictMode is enabled on API 30. */
        if (Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR1) {
            DisplayManager displayManager = (DisplayManager) context.getSystemService(Context.DISPLAY_SERVICE);
            defaultDisplay = displayManager.getDisplay(Display.DEFAULT_DISPLAY);
            defaultDisplay.getRealMetrics(displayMetrics);
        } else {
            defaultDisplay = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE))
                    .getDefaultDisplay();
            defaultDisplay.getMetrics(displayMetrics);
        }
        return displayMetrics;
    }

    public static String getScreenOrientation(Context context) {
        if (context.getResources().getConfiguration().orientation == Configuration
                .ORIENTATION_LANDSCAPE)
            return "landscape";
        else
            return "portrait";
    }

    public static String getSdkVersion() {
        return BuildConfig.SDK_VERSION;
    }

    public static String getLocale(Context context) {
        LocaleListCompat locales = ConfigurationCompat.getLocales(context.getResources().getConfiguration());
        if (locales.size() > 0) {
            return ConfigurationCompat.getLocales(context.getResources().getConfiguration()).get(0).toString();
        }
        return context.getResources().getConfiguration().locale.toString();
    }

    public static long getActiveSessionDuration() {
        return SessionManager.getInstance().getCurrentSessionTimeUntilNow();
    }

    public static String getDeviceArchitecture() {
        return Build.CPU_ABI;
    }

    public static int getOSVersion() {
        return Build.VERSION.SDK_INT;
    }
}
