package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.widget.ImageView;

import java.lang.ref.WeakReference;

import androidx.annotation.DrawableRes;
import androidx.annotation.IdRes;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.R;

/**
 * Created by Hossam on 10/10/2016.
 */

public class BitmapWorkerTask extends AsyncTask<String, Void, Bitmap> {
    private final WeakReference<ImageView> imageViewReference;
    private float targetWidth;
    private float targetHeight;
    private boolean resize;
    @DrawableRes
    int fallbackResourceDrawable;
    @Nullable
    private OnImageLoadedListener onImageLoadedListener;

    public BitmapWorkerTask(final ImageView imageView) {
        // Use a WeakReference to ensure the ImageView can be garbage collected
        imageViewReference = new WeakReference<>(imageView);
    }

    public BitmapWorkerTask(final ImageView imageView, @DrawableRes int fallbackResourceDrawable) {
        // Use a WeakReference to ensure the ImageView can be garbage collected
        imageViewReference = new WeakReference<>(imageView);
        this.fallbackResourceDrawable = fallbackResourceDrawable;
    }

    public BitmapWorkerTask(final ImageView imageView, float targetWidth, float targetHeight) {
        this(imageView);
        this.targetWidth = targetWidth;
        this.targetHeight = targetHeight;
        resize = true;
    }

    public BitmapWorkerTask(final ImageView imageView, @Nullable OnImageLoadedListener onImageLoadedListener) {
        this(imageView);
        this.onImageLoadedListener = onImageLoadedListener;
    }

    public BitmapWorkerTask(final ImageView imageView, float targetWidth, float targetHeight,
                            @Nullable OnImageLoadedListener onImageLoadedListener) {
        this(imageView, targetWidth, targetHeight);
        this.onImageLoadedListener = onImageLoadedListener;
    }

    // Decode image in background.
    @Override
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    protected Bitmap doInBackground(String... params) {
        Bitmap bitmap = null;
        try {
            bitmap = BitmapUtils.decodeSampledBitmapFromLocalPath(params[0]);
            if (resize) {
                bitmap = BitmapUtils.resizeBitmap(bitmap, targetWidth, targetHeight);
            }
        } catch (Exception | OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while loading bitmap", e);
            return null;
        }
        return bitmap;
    }

    // Once complete, see if ImageView is still around and set bitmap.
    @Override
    protected void onPostExecute(@Nullable Bitmap bitmap) {
        final ImageView imageView = imageViewReference.get();
        if (bitmap != null) {
            if (imageView != null) {
                imageView.setImageBitmap(bitmap);
                if (onImageLoadedListener != null) {
                    onImageLoadedListener.onImageLoaded();
                }
            }
        } else {
            if (imageView != null) {
                try {
                    imageView.setImageResource(fallbackResourceDrawable);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while loading image resource", e);
                }
            }
        }
    }

    public interface OnImageLoadedListener {
        void onImageLoaded();
    }

}