package com.instabug.library.user;

import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.model.UserAttribute;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

public class UserAttributeManager {

    private static UserAttributeManager userAttributeManagerInstance;

    private UserAttributeManager() {
    }

    public static UserAttributeManager getInstance() {
        if (userAttributeManagerInstance == null) {
            userAttributeManagerInstance = new UserAttributeManager();
        }
        return userAttributeManagerInstance;
    }

    /**
     * on login event sent it will check first
     * if there's any user attributes for the previous anonymous user
     * - yes then merge it and delete that anonymous user's attributes.
     * - no then do nothing.
     *
     * @param uuid of identified user
     */
    public void migrateUserAttributes(final String uuid) {
        List<UserAttribute> anonymousUserAttributeList = UserAttributeCacheManager.retrieveAnonymousUserAttributes();
        if (!anonymousUserAttributeList.isEmpty()) {
            mergeUserAttribute(anonymousUserAttributeList, uuid);
            UserAttributeCacheManager.deleteAnonymousUserAttribute();
        }
    }

    /**
     * get user attribute and merge it
     *
     * @param anonymousUserAttributeList user attribute saved when the user was anonymous.
     * @param uuid                       user identifier
     */
    private void mergeUserAttribute(List<UserAttribute> anonymousUserAttributeList, String uuid) {
        HashMap<String, String> identifiedUserAttributes = UserAttributeCacheManager.retrieveAll();
        updateUserAttributeForExistingUser(anonymousUserAttributeList, identifiedUserAttributes);
        List<UserAttribute> newUserAttributeList = mapUserAttributeListToHashMap(uuid, identifiedUserAttributes);
        UserAttributeCacheManager.insertAll(newUserAttributeList);
    }

    /**
     * This method to update existing user attribute for the identified user with user's session
     * whilst being anonymous
     *
     * @param anonymousUserAttribute user attribute list in anonymous mode
     * @param existingUserAttributes identified user attribute
     */
    @VisibleForTesting
    void updateUserAttributeForExistingUser(
            @NonNull List<UserAttribute> anonymousUserAttribute,
            @NonNull HashMap<String, String> existingUserAttributes) {

        for (UserAttribute userAttribute : anonymousUserAttribute) {
            existingUserAttributes.put(userAttribute.getKey(), userAttribute.getValue());
        }

    }

    /**
     * this method to convert hashmap list into array list to be inserted later
     *
     * @param uuid                   user identifier
     * @param existingUserAttributes user attribute hashmap to get an array list of user attribute from it.
     * @return new user attribute list filled with the provided list
     */
    @VisibleForTesting
    List<UserAttribute> mapUserAttributeListToHashMap(String uuid,
                                                      @NonNull HashMap<String, String> existingUserAttributes) {
        List<UserAttribute> userAttributeList = new ArrayList<>();
        for (Map.Entry<String, String> entry : existingUserAttributes.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            UserAttribute userAttribute = new UserAttribute.Builder(key, value)
                    .type(UserAttribute.Type.SDK)
                    .anonymous(false)
                    .uuid(uuid)
                    .build();
            userAttributeList.add(userAttribute);
        }
        return userAttributeList;
    }

}
