package com.instabug.library.sessionreplay

import com.instabug.library.InstabugNetworkJob
import com.instabug.library.SessionSyncListener
import com.instabug.library.core.plugin.PluginsManager
import com.instabug.library.model.v3Session.IBGSession
import com.instabug.library.model.v3Session.IBGSessions
import com.instabug.library.model.v3Session.RatingDialogDetection.REVIEW_KEYBOARD_DURATION_KEY
import com.instabug.library.model.v3Session.RatingDialogDetection.REVIEW_PROMPT_DURATION_KEY
import com.instabug.library.sessionV3.cache.SessionCacheManager
import com.instabug.library.sessionV3.manager.IBGSessionManager
import com.instabug.library.sessionV3.providers.FeatureSessionDataController
import com.instabug.library.sessionreplay.configurations.ColdLaunchProvider
import com.instabug.library.sessionreplay.model.SRData
import com.instabug.library.sessionreplay.model.SessionMetadata
import com.instabug.library.sessionreplay.monitoring.SRAnalytics
import com.instabug.library.sessionreplay.monitoring.SRMonitoringSpansDataStore
import com.instabug.library.util.extenstions.logVerbose
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.util.threading.schedule
import org.json.JSONObject
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit

fun interface SessionReplayUserEvaluator {
    fun evaluate()
}

class SessionReplayUserEvaluatorImpl(
    private val sessionsCacheManager: SessionCacheManager,
    private val sessionSyncListener: SessionSyncListener,
    private val metadataHandler: SRMetadataDBHandler,
    private val filesDirectory: OperableSpansCacheDirectory<SRSessionDirectory>,
    private val deleteSessionOperationGetter: (String) -> DeleteSessionDirectoryOperation,
    private val analyticsDataStore: SRMonitoringSpansDataStore,
    private val v3NetworkJob: InstabugNetworkJob
) : SessionReplayUserEvaluator {
    override fun evaluate() {
        val allSessionsMetadata = metadataHandler.queryAll().associateBy { it.uuid }
        val sessionsToEvaluate = sessionsCacheManager.querySessionsBySrEvaluated(false)
            .filter { it.id != IBGSessionManager.currentSession?.id }

        markSessionWithNoSRAsEvaluated(sessionsToEvaluate, allSessionsMetadata)

        sessionsToEvaluate
            .asSequence()
            .filter { session ->
                val metadata = allSessionsMetadata[session.id]
                metadata != null && metadata.isNotRunning()
            }
            .onEach { session -> evaluate(session.id, session.toMetadata()) }
            .forEach { session -> sessionsCacheManager.updateSrEvaluated(session.id, true) }
            .also { v3NetworkJob.start() }
    }

    private fun markSessionWithNoSRAsEvaluated(
        sessionsV3: IBGSessions,
        allSessionsMetadata: Map<String, SRSessionMetadata>
    ) {
        sessionsV3
            .filter { allSessionsMetadata[it.id] == null }
            .forEach { sessionsCacheManager.updateSrEvaluated(it.id, true) }
    }

    private fun SRSessionMetadata?.isNotRunning() = this?.status != SyncStatus.RUNNING

    private fun evaluate(sessionid: String, sessionMetadata: SessionMetadata) {
        val shouldBeSent = sessionSyncListener.onSessionReadyToSync(sessionMetadata)
        if (!shouldBeSent) deleteSessionReplay(sessionid)
    }

    private fun deleteSessionReplay(sessionId: String) {
        "the Session Replay with ID $sessionId is rejected by the user callback"
            .logVerbose(SR_LOG_TAG)
        deleteSessionOperationGetter(sessionId).also(filesDirectory::operate)
        metadataHandler.delete(sessionId)
        sessionsCacheManager.disableSessionsSR(sessionId)
        analyticsDataStore.storeImmediately(SRAnalytics(sessionId = sessionId, isSDKSampled = true))

    }

    private fun IBGSession.toMetadata(): SessionMetadata {
        val hasAppReview = this.ratingDialogDetection
            ?.let(::JSONObject)
            ?.let(::isLinkedToAppReview)
            ?: false

        val builder = SessionMetadata.Builder(
            appVersion = appData.appVersion.orEmpty(),
            device = appData.device,
            linkedToReview = hasAppReview,
            os = appData.os,
            sessionDurationInSeconds = TimeUnit.MICROSECONDS.toSeconds(durationInMicro)
        )
        PluginsManager.getFeaturesSessionDataControllers()
            .mapNotNull { dataController -> collectSessionReplayData(dataController) }
            .map { future -> future.get() }
            .onEach { metaDataCustomizer -> metaDataCustomizer?.customize(builder) }
        return builder.build()
    }

    private fun isLinkedToAppReview(appReviewJson: JSONObject): Boolean {
        val reviewShown = appReviewJson.optLong(REVIEW_PROMPT_DURATION_KEY, 0)
            .let(TimeUnit.MICROSECONDS::toSeconds)
        val keyboardDuration = appReviewJson.optLong(REVIEW_KEYBOARD_DURATION_KEY, 0)
            .let(TimeUnit.MICROSECONDS::toSeconds)
        return reviewShown > 0 && keyboardDuration > 2
    }

    private fun IBGSession.collectSessionReplayData(dataController: FeatureSessionDataController): Future<SRData?>? =
        PoolProvider.submitIOTask { dataController.collectSessionReplayData(this.id) }


}

class RNSessionReplayUserEvaluatorImpl(
    private val sessionReplayUserEvaluator: SessionReplayUserEvaluator,
    private val coldLaunchProvider: ColdLaunchProvider,
    private val poolProvider: PoolProvider,
    private val timeToDelayInMillis: Long = RN_USER_EVALUATION_TIME_MILLIS,
) :
    SessionReplayUserEvaluator {
    override fun evaluate() = poolProvider
        .takeIf { coldLaunchProvider.isColdAPPLaunch }
        ?.schedule(timeToDelayInMillis, sessionReplayUserEvaluator::evaluate)
        ?.also { coldLaunchProvider.isColdAPPLaunch = false }
        ?: sessionReplayUserEvaluator.evaluate()

}