package com.instabug.library.sessionreplay

import androidx.annotation.StringDef
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionReplayMetadataEntry
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.internal.storage.cache.dbv2.WhereClause
import com.instabug.library.internal.storage.cache.dbv2.asArgs
import com.instabug.library.internal.storage.cache.dbv2.joinToArgs
import com.instabug.library.internal.storage.cache.dbv2.kDelete
import com.instabug.library.internal.storage.cache.dbv2.kQuery
import com.instabug.library.sessionreplay.SyncStatus.Companion.OFFLINE
import com.instabug.library.sessionreplay.SyncStatus.Companion.READY_FOR_SCREENSHOTS_SYNC
import com.instabug.library.sessionreplay.SyncStatus.Companion.READY_FOR_SYNC
import com.instabug.library.sessionreplay.SyncStatus.Companion.RUNNING
import com.instabug.library.sessionreplay.SyncStatus.Companion.SYNCED
import com.instabug.library.util.extenstions.getLong
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.getString
import com.instabug.library.util.extenstions.runOrLogAndReport

@Retention(AnnotationRetention.SOURCE)
@StringDef(value = [RUNNING, OFFLINE, READY_FOR_SYNC, READY_FOR_SCREENSHOTS_SYNC, SYNCED])
annotation class SyncStatus {
    companion object {
        const val RUNNING = "RUNNING"
        const val OFFLINE = "OFFLINE"
        const val READY_FOR_SYNC = "READY_FOR_SYNC"
        const val READY_FOR_SCREENSHOTS_SYNC = "READY_FOR_SCREENSHOTS_SYNC"
        const val SYNCED = "SYNCED"
    }
}

data class SRSessionMetadata(
    val uuid: String,
    val startTime: Long,
    val partialId: UInt,
    @SyncStatus var status: String,
)

interface SRMetadataDBHandler {
    fun insert(metadata: SRSessionMetadata)
    fun queryAll(): List<SRSessionMetadata>
    fun queryByStatus(@SyncStatus vararg statuses: String): List<SRSessionMetadata>
    fun updateStatus(uuid: String, @SyncStatus status: String)
    fun updateSyncStatus(uuids: List<String>, @SyncStatus from: String, @SyncStatus to: String)
    fun delete(uuid: String)
}

private const val FAILED_TO_QUERY_SR_SESSIONS = "Failed to query SR sessions"

class DefaultSRMetadataDBHandler(private val databaseManager: IBGDbManager?) : SRMetadataDBHandler {

    override fun insert(metadata: SRSessionMetadata) {
        runCatching {
            databaseManager?.insert(
                SessionReplayMetadataEntry.TABLE_NAME,
                null,
                metadata.toContentValues()
            )
        }.runOrLogAndReport("Failed to insert SR session metadata")
    }

    override fun queryAll(): List<SRSessionMetadata> = runCatching {
        databaseManager?.kQuery(
            table = SessionReplayMetadataEntry.TABLE_NAME
        )?.use(this::toSRSessionMetadataList) ?: emptyList()
    }.getOrLogAndReport(emptyList(), FAILED_TO_QUERY_SR_SESSIONS)

    override fun queryByStatus(@SyncStatus vararg statuses: String): List<SRSessionMetadata> =
        runCatching {
            databaseManager?.kQuery(
                table = SessionReplayMetadataEntry.TABLE_NAME,
                whereClause = whereSyncStatusIs(*statuses)
            )?.use(this::toSRSessionMetadataList) ?: emptyList()
        }.getOrLogAndReport(emptyList(), "Failed to query SR sessions metadata by status")

    override fun updateStatus(uuid: String, @SyncStatus status: String) {
        runCatching {
            val values = IBGContentValues().apply {
                put(SessionReplayMetadataEntry.COLUMN_STATUS, status, true)
            }
            databaseManager?.update(
                SessionReplayMetadataEntry.TABLE_NAME,
                values,
                "${SessionReplayMetadataEntry.COLUMN_ID} = ?",
                listOf(IBGWhereArg(uuid, true))
            )
        }.runOrLogAndReport("Failed to update SR session metadata status")
    }

    override fun updateSyncStatus(
        uuids: List<String>,
        @SyncStatus from: String,
        @SyncStatus to: String
    ) {
        runCatching {
            val values = IBGContentValues().apply {
                put(SessionReplayMetadataEntry.COLUMN_STATUS, to, true)
            }
            val whereClause =
                "${SessionReplayMetadataEntry.COLUMN_ID} in ${uuids.joinToArgs()} and ${SessionReplayMetadataEntry.COLUMN_STATUS} = ?"
            databaseManager?.update(
                SessionReplayMetadataEntry.TABLE_NAME,
                values,
                whereClause,
                uuids.map { IBGWhereArg(it, true) }.toMutableList()
                    .also { it.add(IBGWhereArg(from, true)) })
        }.runOrLogAndReport("Something went wrong when updating v3 sync status")
    }

    override fun delete(uuid: String) {
        runCatching {
            databaseManager?.kDelete(
                table = SessionReplayMetadataEntry.TABLE_NAME,
                selection = "${SessionReplayMetadataEntry.COLUMN_ID} = ?",
                selectionArgs = listOf(IBGWhereArg(uuid, true))
            )
        }.runOrLogAndReport("Failed to delete SR session metadata")
    }

    private fun SRSessionMetadata.toContentValues() = IBGContentValues().apply {
        put(SessionReplayMetadataEntry.COLUMN_ID, uuid, true)
        put(SessionReplayMetadataEntry.COLUMN_START_TIME, startTime, true)
        put(SessionReplayMetadataEntry.COLUMN_PARTIAL_ID, partialId.toLong(), true)
        put(SessionReplayMetadataEntry.COLUMN_STATUS, status, true)
    }

    private fun IBGCursor.toSRSessionMetadata() = SRSessionMetadata(
        uuid = getString(SessionReplayMetadataEntry.COLUMN_ID),
        startTime = getLong(SessionReplayMetadataEntry.COLUMN_START_TIME),
        partialId = getLong(SessionReplayMetadataEntry.COLUMN_PARTIAL_ID).toUInt(),
        status = getString(SessionReplayMetadataEntry.COLUMN_STATUS)
    )

    private fun toSRSessionMetadataList(cursor: IBGCursor): List<SRSessionMetadata> = with(cursor) {
        val resultList = mutableListOf<SRSessionMetadata>()
        while (moveToNext()) resultList.add(toSRSessionMetadata())
        return resultList
    }

    private fun whereSyncStatusIs(@SyncStatus vararg statuses: String): WhereClause =
        statuses.toList().let { statusesList ->
            "${SessionReplayMetadataEntry.COLUMN_STATUS} IN ${statusesList.joinToArgs()}" to statusesList.asArgs()
        }
}