package com.instabug.library.sessionV3.ratingDialogDetection

import android.annotation.SuppressLint
import android.app.Activity
import android.os.Build
import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.model.v3Session.RatingDialogDetection.PLAY_STORE_ACTIVITY
import com.instabug.library.model.v3Session.RatingDialogDetection.RATING_DIALOG_DETECTOR_END_ERROR
import com.instabug.library.model.v3Session.RatingDialogDetection.RATING_DIALOG_DETECTOR_INIT_ERROR
import com.instabug.library.model.v3Session.SessionEvent
import com.instabug.library.sessionV3.configurations.IBGSessionConfigurations
import com.instabug.library.sessionV3.configurations.RatingDialogDetectionConfigurations
import com.instabug.library.sessionV3.manager.IBGSessionManager
import com.instabug.library.sessionV3.providers.IBGSessionDataProvider
import com.instabug.library.util.DeviceStateProvider
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.Executor

data class RatingDialogData(
    val endTimeStampMicros: Long,
    val dialogDurationMicros: Long,
    val keyboardDurationMicros: Long?
)

interface RatingDialogDetector {

    /**
     * @param activity
     * registering keyboard listener here if the activity is google play activity
     */
    fun init(activity: Activity)

    /**
     * @param screenName to validate that it's a google play activity
     * detect start time for rating dialog in micros
     */
    fun startDetection(screenName: String?)

    /**
     * @param screenName to validate that it's a google play activity
     * mark rating dialog ready to be cached
     */
    fun endDetection(screenName: String?)
}

const val RATING_DIALOG_EXECUTOR = "RATING_DIALOG_EXECUTOR"

class RatingDialogDetectorImpl(
    private val sessionExecutor: Executor,
    private val sessionDataProvider: IBGSessionDataProvider,
    private val keyboardDurationDetector: KeyboardDurationDetector,
    private val ratingDialogDetectionConfigs: RatingDialogDetectionConfigurations,
    private val sessionConfigurations: IBGSessionConfigurations
) : RatingDialogDetector {
    private val currentTimeMicros: Long
        get() = System.nanoTime() / 1000

    @VisibleForTesting
    var startTimeMicros: Long? = null

    @VisibleForTesting
    var endTimeMicros: Long? = null

    @VisibleForTesting
    var endTimeStampMicros: Long? = null

    private fun isFeatureEnabled() =
        ratingDialogDetectionConfigs.isEnabled && sessionConfigurations.isV3SessionEnabled

    private fun checkOnIOThenRunOnMain(task: () -> Unit) {
        PoolProvider.postOrderedIOTask(RATING_DIALOG_EXECUTOR) {
            if (isFeatureEnabled()) {
                PoolProvider.postMainThreadTask(task)
            }
        }
    }

    private fun checkOnIOThenExecute(task: () -> Unit) {
        PoolProvider.postOrderedIOTask(RATING_DIALOG_EXECUTOR) {
            if (isFeatureEnabled()) {
                task.invoke()
            }
        }
    }

    @SuppressLint("NewApi")
    override fun init(activity: Activity) = sessionExecutor.execute {
        if (isPlayStoreRatingScreen(activity::class.simpleName))
            checkOnIOThenRunOnMain {
                runOrLogError(errorMessage = RATING_DIALOG_DETECTOR_INIT_ERROR) {
                    if (DeviceStateProvider.getOSVersion() >= Build.VERSION_CODES.R) {
                        keyboardDurationDetector.register(activity)
                    } else {
                        InstabugSDKLogger.d(
                            Constants.LOG_TAG,
                            "Skipping keyboard duration detection"
                        )
                    }
                }
            }

    }

    override fun startDetection(screenName: String?) {
        val startTimeMicros = currentTimeMicros
        if (isPlayStoreRatingScreen(screenName)) {
            checkOnIOThenExecute {
                sessionExecutor.execute { this.startTimeMicros = startTimeMicros }
            }
        }

    }

    override fun endDetection(screenName: String?) {
        val endTimeMicros = currentTimeMicros
        val endTimeStampMicros = System.currentTimeMillis() * 1000
        if (isPlayStoreRatingScreen(screenName)) {
            checkOnIOThenExecute {
                sessionExecutor.execute {
                    runOrLogError(errorMessage = RATING_DIALOG_DETECTOR_END_ERROR) {
                        this.endTimeMicros = endTimeMicros
                        this.endTimeStampMicros = endTimeStampMicros
                        postRatingDialogDataIfPossible()
                        clearAndReset()
                    }
                }
            }
        }
    }

    @SuppressLint("NewApi")
    private fun postRatingDialogDataIfPossible() {
        val duration = (endTimeMicros ?: 0) - (startTimeMicros ?: 0)
        val endTimeStamp = endTimeStampMicros ?: 0
        if (isReadyForCache()) {
            val keyboardDuration: Long? =
                if (DeviceStateProvider.getOSVersion() >= Build.VERSION_CODES.R) {
                    keyboardDurationDetector.getDuration()
                } else 0

            val ratingDialogData =
                RatingDialogData(endTimeStamp, duration, keyboardDuration)
            IBGSessionManager.emit(
                sessionEvent = SessionEvent.RatingDialogDataReady(ratingDialogData),
                handleEventOnSameThread = true
            )
        }
    }


    @SuppressLint("NewApi")
    private fun clearAndReset() {
        if (DeviceStateProvider.getOSVersion() >= Build.VERSION_CODES.R) {
            keyboardDurationDetector.unregister()
        }
        startTimeMicros = null
        endTimeMicros = null
        endTimeStampMicros = null
    }

    private fun isReadyForCache() = sessionDataProvider.isInstalledFromPlayStore
            && isValidTime(startTimeMicros)
            && isValidTime(endTimeMicros)
            && isValidTime(endTimeStampMicros)
            && (endTimeMicros ?: 0) > (startTimeMicros ?: 0)

    private fun isValidTime(time: Long?) = time != null && time > 0

    private fun isPlayStoreRatingScreen(screenName: String?) = screenName
        ?.let { it.trim() == PLAY_STORE_ACTIVITY } ?: false
}