package com.instabug.library.sessionV3.providers

import android.annotation.SuppressLint
import android.content.Context
import com.instabug.library.IBGFeature
import com.instabug.library.Instabug
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.device.InstabugDeviceProperties
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper
import com.instabug.library.logging.InstabugUserEventLogger
import com.instabug.library.model.UserAttributes
import com.instabug.library.model.session.config.SyncMode
import com.instabug.library.model.v3Session.Constants.DEBUG_PROPERTY
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STORE_URL
import com.instabug.library.model.v3Session.RatingDialogDetection
import com.instabug.library.model.v3Session.StartTime
import com.instabug.library.model.v3Session.StitchingState
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tokenmapping.TokenMappingServiceLocator
import com.instabug.library.user.UserEvent
import com.instabug.library.user.UserManager
import com.instabug.library.util.AppUtils
import com.instabug.library.util.DeviceStateProvider
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.filters.AttributeFiltersFunctions
import com.instabug.library.util.filters.Filters
import java.util.Random

interface SessionAppDataProvider {
    val os: String
    val device: String
    val appVersion: String?
    val appToken: String?
    val sdkVersion: String?
    val isInDebugMode: Boolean
    val locale: String?
    val screenSize: String?
    val isInstalledFromPlayStore: Boolean
}

interface SessionUserDataProvider {
    val uuid: String
    val usersPageEnabled: Boolean
    val userName: String?
    val userEmail: String?
    val userEvents: List<UserEvent>
    val userAttributes: UserAttributes?
    val UserAttributes?.asStringJson: String
    val UserAttributes?.keysAsStringJsonArray: String
    val List<UserEvent>.asStringJsonArray: String
    val List<UserEvent>.keysAsStringJsonArray: String
    val randomID: UInt
}

interface SessionProductionUsageProvider {

    val storeURL: String
    val bugsEnabled: Boolean
    val surveysEnabled: Boolean
    val featureRequestEnabled: Boolean
    val apmEnabled: Boolean
    val crashesEnabled: Boolean
}

interface IBGSessionDataProvider : SessionAppDataProvider,
    SessionUserDataProvider,
    SessionProductionUsageProvider {
    val isV2SessionSent: Boolean
    fun getStitchingState(startTime: StartTime): StitchingState
}

object SessionDataProvider : IBGSessionDataProvider {
    //region AppData
    private val context: Context?
        get() = Instabug.getApplicationContext()

    override val os
        get() = DeviceStateProvider.getOS()


    override val device
        get() = if (InstabugDeviceProperties.isProbablyAnEmulator())
            "Emulator - " + InstabugDeviceProperties.getDeviceType()
        else InstabugDeviceProperties.getDeviceType()

    override val appVersion: String?
        get() = context?.let { DeviceStateProvider.getAppVersion(it) }

    override val appToken
        get() = TokenMappingServiceLocator.getTokenMappingConfigs().availableAppToken

    override val sdkVersion: String?
        get() = InstabugCore.getSDKVersion()

    override val isInDebugMode: Boolean
        get() = context?.applicationContext?.packageName
            ?.takeUnless { it.isEmpty() }
            ?.let { packageName -> packageName == getProp(DEBUG_PROPERTY) }
            ?: false
    override val locale: String?
        get() = DeviceStateProvider.getLocale(context)
    override val screenSize: String?
        get() = DeviceStateProvider.getScreenSize(context)
    override val isInstalledFromPlayStore: Boolean
        get() = storeURL.trim() == RatingDialogDetection.PLAY_STORE_PACKAGE_NAME

    @SuppressLint("PrivateApi")
    fun getProp(key: String?): String? =
        runOrLogError { Class.forName("android.os.SystemProperties") }
            .getOrNull()
            ?.runOrLogError { getDeclaredMethod("get", String::class.java) }
            ?.getOrNull()
            ?.runOrLogError { invoke(null, key) as String }
            ?.getOrNull()


    //endregion

    //region UserData
    override val uuid: String
        get() = UserManager.getUUID()
    override val usersPageEnabled
        get() = InstabugCore.isUsersPageEnabled()
    override val userName
        get() = UserManager.identifiedUsername
    override val userEmail
        get() = UserManager.identifiedUserEmail

    override val userEvents: List<UserEvent>
        get() = InstabugUserEventLogger.getInstance()
            .userEvents
            .toList()

    override val userAttributes
        get() = Filters.applyOn(UserAttributesDbHelper.getAll())
            .apply(AttributeFiltersFunctions.nonBackEndMapFilter())
            .thenGet()
            ?.takeUnless { it.isEmpty() }
            ?.let { attrs -> UserAttributes().apply { putMap(attrs) } }

    override val UserAttributes?.asStringJson: String
        get() = this?.toString() ?: "{}"

    override val UserAttributes?.keysAsStringJsonArray: String
        get() = this?.keysAsStringJsonArray() ?: "[]"

    override val List<UserEvent>.asStringJsonArray: String
        get() = runOrLogError(errorMessage = "parsing user events got error: ") {
            UserEvent.toJson(this).toString()
        }.getOrElse { "[]" }
    override val randomID: UInt
            get() = Random().nextInt().toUInt()

    override val List<UserEvent>.keysAsStringJsonArray: String
        get() = run(UserEvent::keysAsJsonArray).toString()

    //endregion

    //region Production usage

    private val String.isEnabled: Boolean
        get() = InstabugCore.isFeatureEnabled(this)
    override val storeURL: String
        get() = context?.let(AppUtils::getInstallerPackageName)
            ?: DEFAULT_STORE_URL
    override val bugsEnabled: Boolean
        get() = IBGFeature.BUG_REPORTING.isEnabled
    override val surveysEnabled: Boolean
        get() = IBGFeature.SURVEYS.isEnabled
    override val featureRequestEnabled: Boolean
        get() = IBGFeature.FEATURE_REQUESTS.isEnabled
    override val apmEnabled: Boolean
        get() = InstabugCore.isAPMEnabled()
    override val crashesEnabled: Boolean
        get() = InstabugCore.isCrashReportingEnabled()
    override val isV2SessionSent: Boolean
        get() = context
            ?.let(SettingsManager::getSessionsSyncConfigurations)
            .let { config -> config?.syncMode ?: SyncMode.OFF }
            .let { mode -> mode == SyncMode.BATCHING || mode == SyncMode.NO_BATCHING }


    //endregion

    //region Session

    override fun getStitchingState(startTime: StartTime): StitchingState =
        if (startTime.isBackground)
            StitchingState.BACKGROUND_SESSION
        else
            IBGSessionServiceLocator.stitchingManger.isSessionLead(startTime.foregroundMicroStartTime)
                .let { isStitchingLead ->
                    if (isStitchingLead)
                        StitchingState.SESSION_LEAD
                    else
                        StitchingState.STITCHED
                }

    //endregion
}