package com.instabug.library.sessionV3.providers

import androidx.annotation.WorkerThread
import com.instabug.library.model.v3Session.IBGSessionData
import com.instabug.library.sessionreplay.model.SRData

typealias SessionID = String

/**
 * a session data controller that will control session data in feature module
 * this will collect session data by sessions ids and return the feature data
 * that will be included in the session request
 * and handle session
 * */
interface FeatureSessionDataController {
    /**
     * collects sessions data from features
     * @param sessionsIds a list of sessions ids that will be sent to BackEnd
     *@return map of SessionData that will be included in the session request
     * */
    @WorkerThread
    fun collectSessionsData(sessionsIds: List<SessionID>): Map<SessionID, IBGSessionData>

    /**
     * Collects session replay data for the specified session IDs.
     *
     * This function retrieves relevant session replay data for the provided session ID.
     * The data is returned as an instance of [SRData] containing the corresponding session replay information.
     *
     * @param sessionId The id of the session for which to collect data.
     * @return An instance of [SRData] containing session replay data for the provided session id.
     * Or Null if the Feature is not interested in adding data to Session replay
     */
    @WorkerThread
    fun collectSessionReplayData(sessionId: SessionID): SRData? = null

    /**
     *
     * this will be called after sessions sent successfully
     * or sessions reached the store limit to drop sessions data in other features
     * @param sessionsIds a list of sessions ids that no longer needs its data
     * */
    @WorkerThread
    fun dropSessionData(sessionsIds: List<SessionID>)
}

/**
 * this will be implemented by features that provide data to be included in the session request
 *
 * */
interface FeatureSessionDataControllerHost {
    /**
     * returns feature session data controller
     *@see FeatureSessionDataController
     * */
    val sessionDataController: FeatureSessionDataController
}