package com.instabug.library.session;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_SYNC_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_V2_SESSION_SENT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.TABLE_NAME;
import static com.instabug.library.model.session.SyncStatus.OFFLINE;
import static com.instabug.library.model.session.SyncStatus.READY_FOR_SYNC;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.model.session.SessionLocalEntity;
import com.instabug.library.model.session.SessionMapper;
import com.instabug.library.model.session.SyncStatus;

import java.util.ArrayList;
import java.util.List;

import io.reactivexport.Completable;
import io.reactivexport.CompletableEmitter;
import io.reactivexport.CompletableOnSubscribe;

public class SessionsLocalDataSource {


    public Completable saveOrUpdate(@NonNull final SessionLocalEntity entity) {
        return Completable.create(new CompletableOnSubscribe() {
            @Override
            public void subscribe(@NonNull CompletableEmitter emitter) {
                ContentValues contentValues = SessionMapper.toContentValues(entity);
                SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
                try {
                    db.beginTransaction();
                    if (db.insertWithOnConflict(TABLE_NAME, null, contentValues) == -1) {
                        String whereClause = COLUMN_ID + " = ? ";
                        String[] whereArgs = new String[]{String.valueOf(entity.getId())};
                        db.update(TABLE_NAME, contentValues, whereClause, whereArgs);
                    }
                    db.setTransactionSuccessful();
                } finally {
                    db.endTransaction();
                    db.close();
                }
                emitter.onComplete();
            }
        });
    }

    @WorkerThread
    public SessionsLocalDataSource markAsSyncedWithRemote(@NonNull final List<String> ids) {
        for (String id : ids)
            markAsSyncedWithRemote(id);
        return this;
    }

    public void markAsSyncedWithRemote(final String id) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_SYNC_STATUS, SyncStatus.SYNCED_WITH_REMOTE);
        try {
            String whereClause = COLUMN_ID + " = ? ";
            String[] whereArgs = new String[]{id};
            db.update(TABLE_NAME, contentValues, whereClause, whereArgs);
        } finally {
            db.close();
        }
    }


    @WorkerThread
    public void markOfflineAsReadyForSync() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_SYNC_STATUS, READY_FOR_SYNC);
        try {
            db.beginTransaction();
            String whereClause = COLUMN_SYNC_STATUS + " = ? ";
            String[] whereArgs = new String[]{String.valueOf(OFFLINE)};
            db.update(TABLE_NAME, contentValues, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public Completable migrateUUID(@NonNull final String oldUUID, @NonNull final String newUUID) {
        return Completable.create(new CompletableOnSubscribe() {
            @Override
            public void subscribe(@NonNull CompletableEmitter emitter) {
                SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_UUID, newUUID);
                String whereClause = COLUMN_UUID + " = ? ";
                String[] whereArgs = new String[]{oldUUID};
                try {
                    db.beginTransaction();
                    db.update(TABLE_NAME, contentValues, whereClause, whereArgs);
                    db.setTransactionSuccessful();
                } finally {
                    db.endTransaction();
                    db.close();
                }
                emitter.onComplete();
            }
        });
    }

    public void delete(@NonNull final List<String> ids) {
        for (String id : ids)
            delete(id);
    }

    public void delete(final String id) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + " = ? ";
        String[] whereArgs = new String[]{id};
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
        } finally {
            db.close();
        }
    }

    public void deleteInvalidSessions() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_V2_SESSION_SENT + " = 0 ";
        try {
            db.delete(TABLE_NAME, whereClause, null);
        } catch (Exception throwable) {
            IBGDiagnostics.reportNonFatal(throwable, "Something Went Wrong While Deleting Invalid Sessions With s2s false");
        }
    }

    public void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @WorkerThread
    @NonNull
    public List<SessionLocalEntity> queryAllReadyForSync() {
        List<SessionLocalEntity> localEntities = new ArrayList<>();
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();
        Cursor cursor = database.query(TABLE_NAME, null, COLUMN_SYNC_STATUS + " =? ",
                new String[]{String.valueOf(READY_FOR_SYNC)}, null, null, null);
        try {
            if (cursor != null && cursor.moveToFirst()) {
                do {
                    localEntities.add(SessionMapper.toLocalEntity(cursor));
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Retrieve ready for sync sessions failed: " + e.getMessage(), Constants.LOG_TAG);
        } finally {
            if (cursor != null) cursor.close();
            database.close();
        }
        return localEntities;
    }
}
