package com.instabug.library.session

import com.instabug.library.Instabug

interface SessionsConfigurationsManager {
    /**
     * Sets last  Sessions request time
     */
    fun setLastSessionsRequestStartedAt(time: Long)

    /**
     * Sets last  Sessions limited until period
     */
    fun setSessionsLimitedUntil(period: Int)

    /**
     * returns true if the current time between request start time
     * and and the limitation period
     */
    fun isSessionsRateLimited(): Boolean
}

object SessionsConfigurationsManagerImpl : SessionsConfigurationsManager {
    private const val LAST_SESSIONS_REQUEST_STARTED_AT = "last_sessions_request_started_at"
    private const val SESSIONS_LIMITED_UNTIL = "sessions_rate_limited_until"
    private val sharedPreferences by lazy {
        Instabug.getApplicationContext()
            ?.let(SessionsServiceLocator::getPreferencesUtils)
    }
    private var lastSessionRequestStartedAt: Long
        get() {
            return sharedPreferences?.getLong(LAST_SESSIONS_REQUEST_STARTED_AT, 0) ?: 0
        }
        set(value) {
            sharedPreferences?.saveOrUpdateLong(LAST_SESSIONS_REQUEST_STARTED_AT, value)
        }
    private var sessionsLimitedUntil: Long
        get() {
            return sharedPreferences?.getLong(SESSIONS_LIMITED_UNTIL, 0) ?: 0
        }
        set(value) {
            sharedPreferences?.saveOrUpdateLong(SESSIONS_LIMITED_UNTIL, value)
        }

    override fun setLastSessionsRequestStartedAt(time: Long) {
        lastSessionRequestStartedAt = time
    }

    override fun setSessionsLimitedUntil(period: Int) {
        sessionsLimitedUntil = period
            .toLong()
            .times(1000)
            .plus(lastSessionRequestStartedAt)

    }

    override fun isSessionsRateLimited(): Boolean {
        val start: Long = lastSessionRequestStartedAt
        val limitedUntil: Long = sessionsLimitedUntil
        val currentTime = System.currentTimeMillis()
        return start != 0L && limitedUntil != 0L &&
                currentTime > start && currentTime < limitedUntil
    }

}