package com.instabug.library.screenshot.analytics

import androidx.annotation.VisibleForTesting
import com.instabug.library.IBGFeature
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.eventpublisher.Subscriber
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.BR_REPRO_STEPS_DISABLED_SDK
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.BR_RSA_DISABLED_BE
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.BR_SCREENSHOTS_DISABLED_SDK
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.CR_REPRO_STEPS_DISABLED_SDK
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.CR_RSA_DISABLED_BE
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes.Companion.CR_SCREENSHOTS_DISABLED_SDK
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import com.instabug.library.visualusersteps.ReproProxyAuthID
import java.util.concurrent.atomic.AtomicInteger

class ProductAnalyticsCollector constructor(
    private val analyticsHolder: ScreenshotsAnalyticsHolder,
    private val reproConfigurations: ReproConfigurationsProvider
) : ScreenshotsAnalyticsHolder, Subscriber<AnalyticsEvent> {
    @VisibleForTesting
    val productErrors: MutableMap<String, AtomicInteger> = mutableMapOf()
    private var previousReproStepsEnabled: Boolean = true
    private var previousReproScreenshotsEnabled: Boolean = true
    private var previousReproScreenshotsAvailable: Boolean = true
    private var previousIsFeatureAvailable: Boolean = true
    override fun onNewEvent(event: AnalyticsEvent) {
        if (event is AnalyticsEvent.ScreenshotEvent.OnConfigurationChanged) {
            handleConfigurations()
        }
        if (event is AnalyticsEvent.ClearScreenShotAnalyticsData) {
            productErrors.clear()
            previousIsFeatureAvailable = true
            previousReproStepsEnabled = true
            previousReproScreenshotsEnabled = true
            previousReproScreenshotsAvailable = true
        }
    }
    fun handleConfigurations() {
        previousIsFeatureAvailable = updateFeatureState(InstabugCore.isFeatureAvailable(IBGFeature.REPRO_STEPS),
            previousIsFeatureAvailable,
            ScreenshotsErrorCodes.REPRO_STEPS_DISABLED_BE
        )
        previousReproStepsEnabled = updateFeatureState(reproConfigurations.isReproStepsEnabledSDK,
            previousReproStepsEnabled,
            adjustReproStepsDisabledSDKErrorCodes(reproConfigurations.reproProxyAuthId)
        )
        if (previousReproStepsEnabled)
            updateFeatureState(
                reproConfigurations.isReproScreenShotsEnabledSDK,
                previousReproScreenshotsEnabled,
                adjustScreenShotDisabledSDKErrorCodes(reproConfigurations.reproProxyAuthId)
            )
        previousReproScreenshotsEnabled = reproConfigurations.isReproScreenShotsEnabledSDK
        previousReproScreenshotsAvailable = updateFeatureState(reproConfigurations.isReproScreenshotsAvailable, previousReproScreenshotsAvailable, adjustRSADisabledBEErrorCodes(reproConfigurations.reproProxyAuthId)
        )
    }


    override fun getAnalytics(): ScreenshotAnalytics {
        val screenshotAnalytics =
            if (reproConfigurations.isReproScreenshotsEnabled) analyticsHolder.getAnalytics()
            else ScreenshotAnalytics()
        screenshotAnalytics.adjustErrorCodes(productErrors)
        return screenshotAnalytics
    }

    private fun updateFeatureState(currentState: Boolean, previousState: Boolean, errorCode: String): Boolean {
        if (!currentState && previousState) {
            productErrors.getOrPut(errorCode) { AtomicInteger() }.incrementAndGet()
        }
        return currentState
    }


    private fun adjustReproStepsDisabledSDKErrorCodes(reproAuthId: Int): String {
        return if (reproAuthId == ReproProxyAuthID.Bugs) {
            BR_REPRO_STEPS_DISABLED_SDK
        } else {
            CR_REPRO_STEPS_DISABLED_SDK
        }
    }

    private fun adjustScreenShotDisabledSDKErrorCodes(reproAuthId: Int): String {
        return if (reproAuthId == ReproProxyAuthID.Bugs) {
            BR_SCREENSHOTS_DISABLED_SDK
        } else {
            CR_SCREENSHOTS_DISABLED_SDK
        }
    }

    private fun adjustRSADisabledBEErrorCodes(reproAuthId: Int): String {
        return if (reproAuthId == ReproProxyAuthID.Bugs) {
            BR_RSA_DISABLED_BE
        } else {
            CR_RSA_DISABLED_BE
        }
    }

}

