package com.instabug.library.networkv2.service.synclogs;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider;
import com.instabug.library.internal.resolver.LoggingSettingResolver;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.logging.disklogs.LogFilesHelper;
import com.instabug.library.logging.disklogs.LoggingFileProvider;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.service.base.BaseScheduler;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.MD5Generator;
import com.instabug.library.util.TaskDebouncer;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * A class which will start syncing  when  logs are summoned
 * ************
 * DEPENDENCIES
 * ************
 * 1. {@link LoggingSettingResolver}
 * 2. {@link ISyncKeyProvider} for testing and mocking these keys
 * 3. {@link DiskUtils} to get the log directory
 * 4. {@link SyncLogService} to do the actual syncing.
 */
public class SyncLogFacade implements Request.Callbacks<String, Exception> {

    private LoggingSettingResolver loggingSettingResolver;
    private static SyncLogFacade syncLogFacade;
    @Nullable
    private String uuid, email;
    private ISyncKeyProvider syncKeyProvider;
    @Nullable
    private SyncLogService syncLogService;
    private String LAST_UPLOADED_AT_KEY = "logs_last_uploaded_at";

    private TaskDebouncer syncingDebouncer = new TaskDebouncer(TimeUnit.SECONDS.toMillis(1));

    private SyncLogFacade() {
        loggingSettingResolver = LoggingSettingResolver.getInstance();
        syncKeyProvider = new SyncLogKeyProvider();

    }

    public static synchronized SyncLogFacade getInstance() {
        if (syncLogFacade == null)
            syncLogFacade = new SyncLogFacade();

        return syncLogFacade;
    }

    public void setUserData(String uuid, String email) {
        this.uuid = uuid;
        this.email = email;
    }

    /**
     * A method to hash uuid to match the one in the backend
     *
     * @param uuid
     * @return hashed uuid
     */
    @Nullable
    private String hashUuidForMatching(@NonNull String uuid) {
        return MD5Generator.generateMD5(syncKeyProvider.getMatchingUuidPrefix()
                + uuid.toLowerCase(Locale.getDefault()) + syncKeyProvider.getMatchingUuidSuffix());
    }

    /**
     * A method to hash email to match the one in the backend
     *
     * @param email
     * @return hashed email
     */
    @VisibleForTesting
    @Nullable
    String hashEmailForMatching(@NonNull String email) {
        return MD5Generator.generateMD5(syncKeyProvider.getMatchingEmailPrefix()
                + email.toLowerCase(Locale.getDefault()) + syncKeyProvider.getMatchingEmailSuffix());
    }


    /**
     * A method to hash uuid to match the one in the backend
     *
     * @param uuid
     * @return hashed uuid
     */
    @Nullable
    private String hashUuidForSyncing(@NonNull String uuid) {
        return MD5Generator.generateMD5(syncKeyProvider.getSyncingUuidPrefix()
                + uuid.toLowerCase(Locale.getDefault()) + syncKeyProvider.getSyncingUuidSuffix());
    }

    /**
     * A method to hash email to match the one in the backend
     *
     * @param email
     * @return hashed email
     */
    @VisibleForTesting
    @Nullable
    String hashEmailForSyncing(@NonNull String email) {
        return MD5Generator.generateMD5(syncKeyProvider.getSyncingEmailPrefix()
                + email.toLowerCase(Locale.getDefault()) + syncKeyProvider.getSyncingEmailSuffix());
    }


    /**
     * A method to run the syncing logic but necessarily it will sync,
     * only if the keys are matching then it will otherwise it will skip
     */
    public void run(Context context, String token) {
        try {
            if (isSyncRequired() && timeToUpload(context)) {
                syncLogService = SyncLogService.getInstance(new NetworkManager(),
                        new SyncLogRequestMapper(), this, new BaseScheduler());
                Pair<String, String> credentials = getCredentials(email, uuid);
                String userEmail = credentials.first;
                String userUuid = credentials.second;
                File insatbugLogDirectory = DiskUtils.getInstabugLogDirectory(LoggingFileProvider.LOGS_DIRECTORY_NAME, context, false);
                if (insatbugLogDirectory == null) return;
                if (insatbugLogDirectory.exists()) {
                    File[] files = insatbugLogDirectory.listFiles();
                    if (files != null) {
                        sync(files, userUuid, userEmail, token);
                    }
                }
            }
        } catch (UnsatisfiedLinkError error) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while syncing logs", error);
            Instabug.disable();
        }
    }

    /**
     * A method to hash email and uuid
     * if email is matching then hash email as email and password,
     * if uuid is matching then hash uuid as email and password.
     * if both are matching then match email as email and password.
     *
     * @param email user email
     * @param uuid  password
     * @return pair of email and password.
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private Pair<String, String> getCredentials(@Nullable String email, @Nullable String uuid) {
        Pair<String, String> credentials = new Pair<>("", "");
        if (email != null && isEmailMatches()) {
            String userEmail = hashEmailForSyncing(email);
            String password = hashUuidForSyncing(email);
            credentials = new Pair<>(userEmail, password);
        } else if (uuid != null && isUUIDMatches()) {
            String userEmail = hashEmailForSyncing(uuid);
            String password = hashUuidForSyncing(uuid);
            credentials = new Pair<>(userEmail, password);
        }
        return credentials;
    }

    @VisibleForTesting
    void sync(final File[] files, @Nullable final String userUuid, @Nullable final String userEmail, final String token) {
        if (syncLogService != null) {
            syncingDebouncer.debounce(new Runnable() {

                @Override
                public void run() {
                    if (syncLogService != null) {
                        try {
                            syncLogService.sync(Arrays.asList(files),
                                    userUuid, userEmail, token);
                        } catch (UnsupportedEncodingException e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "error while syncing logs", e);
                        }
                    }
                }
            });
        }
    }

    @VisibleForTesting
    boolean timeToUpload(Context context) {
        LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
        if (loggingSettings != null) {
            long uploadInterval = loggingSettings.getUploadInterval();
            return (System.currentTimeMillis() - getLastUploadedAt(context)) > TimeUnit.SECONDS.toMillis(uploadInterval);
        }
        return false;
    }

    /**
     * A method to check if syncing is required, by matching both email or uuid.
     *
     * @return true if any matches, false otherwise
     */
    @VisibleForTesting
    boolean isSyncRequired() {
        return isEmailMatches() || isUUIDMatches();
    }

    /**
     * A method to match the email in targted user with the one we have.
     *
     * @return true if emails matched, fal\\se otherwise.
     */
    @VisibleForTesting
    boolean isEmailMatches() {
        if (email != null && hashEmailForMatching(email) == null) {
            return false;
        }
        LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
        if (loggingSettings != null) {
            Set<String> emailSet =
                    loggingSettings.getEmailSet();
            return emailSet != null && email != null && hashEmailForMatching(email) != null
                    && emailSet.contains(hashEmailForMatching(email));
        }
        return false;
    }

    /**
     * A method to match the uuid in targted user with the one we have.
     *
     * @return true if uuid matched, false otherwise.
     */
    @VisibleForTesting
    boolean isUUIDMatches() {
        if (uuid != null && hashUuidForMatching(uuid) == null) {
            return false;
        }
        LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
        if (loggingSettings != null) {
            Set<String> uuidSet =
                    loggingSettings.getUuidSet();
            return uuidSet != null && uuid != null && hashUuidForMatching(uuid) != null &&
                    uuidSet.contains(hashUuidForMatching(uuid));
        }
        return false;
    }


    @VisibleForTesting
    void setLoggingSettingResolver(LoggingSettingResolver loggingSettingResolver) {
        this.loggingSettingResolver = loggingSettingResolver;
    }

    @VisibleForTesting
    void setSyncKeyProvider(ISyncKeyProvider syncKeyProvider) {
        this.syncKeyProvider = syncKeyProvider;
    }

    /**
     * Get last uploaded at
     *
     * @param context to use for getting SharedPreferences
     * @return last fetched at
     */
    @VisibleForTesting
    long getLastUploadedAt(Context context) {
        SharedPreferences sharedPreferences = CoreServiceLocator.getInstabugSharedPreferences(context,
                SettingsManager.INSTABUG_SHARED_PREF_NAME);
        if (sharedPreferences == null) return 0L;
        return sharedPreferences.getLong(LAST_UPLOADED_AT_KEY, 0L);
    }


    /**
     * Set last uploaded at
     *
     * @param uploadedAt last uploaded at
     * @param context    to use for getting SharedPreferences
     */
    @VisibleForTesting
    void setLastUpladedat(long uploadedAt, Context context) {
        SharedPreferences sharedPreferences = CoreServiceLocator.getInstabugSharedPreferences(context,
                SettingsManager.INSTABUG_SHARED_PREF_NAME);
        if (sharedPreferences == null) return;
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putLong(LAST_UPLOADED_AT_KEY, uploadedAt);
        editor.apply();
    }

    @Override
    public void onSucceeded(@Nullable String filePath) {
        Context context = null;
        if (InstabugApplicationProvider.getInstance() != null) {
            context = InstabugApplicationProvider.getInstance().getApplication();
        }
        if (context != null) {
            setLastUpladedat(System.currentTimeMillis(), context);
        }
        if (filePath != null) {
            File file = new File(filePath);

            if (!LogFilesHelper.isTodayFile(file)) {
                try {
                    if (!file.delete()) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't delete disposable file (" + file.getName() + ")");
                    }
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't delete disposable file", e);
                }
            }
        }
    }

    @Override
    public void onFailed(Exception error) {
        InstabugSDKLogger.e(Constants.LOG_TAG, "exception", error);
    }

    public void tearDown() {
        SyncLogService.tearDown();
    }
}
