package com.instabug.library.networkv2.limitation

import com.instabug.library.Constants
import com.instabug.library.networkv2.RateLimitedException
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils

/**
 * A mediator class for applying rate limitation in crashes synchronization jobs
 * @param M the type of incidents the limiter is intended to limit.
 * @property onLimitationApplied the deletion mechanism for that particular incident type.
 * @constructor creates a limiter object with a mechanism of deleting limited incidents.
 */
class RateLimiter<M, in T : RateLimitationSettings>(
    private val settings: T,
    private val onLimitationApplied: ((M) -> Unit),
    private val feature: RateLimitedFeature? = null
) {

    /**
     * Marks the first successful reporting request to reset the limitation period.
     */
    fun reset() {
        settings.setLastRequestStartedAt(0)
        settings.setLimitedUntil(0)
    }

    /**
     * Inspects a resulting throwable from a failed reporting request.
     * If the throwable is of type [RateLimitedException], uses the given limitation period to set
     * a timeframe for the limitation to end and delete the limited incident.
     *
     * @param throwable the throwable resulting from a failed reporting request
     * @param obj the incident that the request was trying to report
     *
     * @return false if the throwable given is not a [RateLimitedException] hence no limitation
     *          applied and true otherwise
     */
    fun inspect(throwable: Throwable, obj: M): Boolean {
        if (throwable !is RateLimitedException) return false
        settings.setLimitedUntil(throwable.period)
        onLimitationApplied(obj)
        logFeatureIsRateLimited()
        return true
    }

    /**
     * Checks if the reporting is already under limitation. If under limitation, the incident
     * is to be deleted.
     *
     * @param obj the incident that the reporter is attempting to report.
     * @return false if the reporting is not under limitation and true otherwise.
     */
    fun applyIfPossible(obj: M): Boolean {
        if (!settings.isRateLimited) {
            settings.setLastRequestStartedAt(TimeUtils.currentTimeMillis())
            return false
        }
        onLimitationApplied(obj)
        logFeatureIsRateLimited()
        return true
    }

    private fun logFeatureIsRateLimited() {
        feature?.let {
            InstabugSDKLogger.d(
                Constants.LOG_TAG,
                RateLimitedException.RATE_LIMIT_REACHED.format(feature.featureName)
            )
        }
    }

    object Factory {
        fun <M> createWithDefaultSettings(
            feature: RateLimitedFeature,
            limitationAction: ((M) -> Unit)
        ): RateLimiter<M, RateLimitationSettingsImpl> = RateLimiter(
            settings = RateLimitationSettingsImpl(feature),
            onLimitationApplied = limitationAction,
            feature = feature
        )
    }
}