package com.instabug.library.networkv2.connection;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;

import com.instabug.library.Constants;
import com.instabug.library.networkv2.Multipart;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.util.InstabugSDKLogger;

import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.net.HttpURLConnection;

/**
 * An implementation class to manage the connection of requests with multipart.
 */
public class MultipartConnectionManager extends InstabugBaseConnectionManagerImpl {


    @SuppressLint("RESOURCE_LEAK")
    @Override
    public HttpURLConnection setupConnection(HttpURLConnection connection, Request request) throws Exception {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Connect to: " + request.getRequestUrlForLogging() + " with multiPart type");
        connection.setRequestProperty(Header.CONNECTION, "Keep-Alive");
        connection.setRequestProperty(Header.CACHE_CONTROL, "no-cache");
        connection.setRequestProperty(Header.CONTENT_ENCODING, "");
        FileToUpload fileToUpload = request.getFileToUpload();
        if (fileToUpload != null) {
            Multipart multipart = buildMultipartRequest(request, connection, fileToUpload);
            multipart.finish();
        }
        return connection;
    }

    @SuppressLint("RESOURCE_LEAK")
    @NotNull
    private Multipart buildMultipartRequest(Request request,
                                            HttpURLConnection connection,
                                            @NonNull FileToUpload fileToUpload)
            throws IOException {
        Multipart multipart = new Multipart(connection);
        for (RequestParameter requestParameter : request.getRequestBodyParameters()) {
            multipart.addFormField(requestParameter.getKey(),
                    requestParameter.getValue().toString()
            );
        }
        multipart.addFilePart(
                fileToUpload.getFilePartName(),
                fileToUpload.getFilePath(),
                fileToUpload.getFileName(),
                fileToUpload.getFileType());
        return multipart;
    }

    @Override
    public String getContentType() {
        return "application/json";
    }

    @Override
    public int getReadTimeOut() {
        return 0;
    }

    @Override
    public int getConnectTimeOut() {
        return 0;
    }

    @Override
    public RequestResponse handleResponse(HttpURLConnection connection, Request request) throws IOException {
        RequestResponse requestResponse = new RequestResponse();
        int responseCode = connection.getResponseCode();
        requestResponse.setResponseCode(responseCode);
        requestResponse.setHeaders(getHeaderFields(connection));
        String response = convertStreamToString(connection.getInputStream());
        requestResponse.setResponseBody(response);
        connection.disconnect();
        return requestResponse;
    }
}
