package com.instabug.library.networkv2.authorization;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.networkv2.request.Constants;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.FileUtils;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;

/**
 * A class to hash the multipart request
 * *************
 * DEPENDECIES
 * *************
 * 1. {@link FileUtils#getFile(String)}
 */
final class MultipartRequestOfficer {

    /**
     * The last bytes to be read from the file.
     */
    @VisibleForTesting
    static final int BYTES_SIZE = 20;

    /**
     * A method to hash only multipart requests first it hash the last {@link MultipartRequestOfficer#BYTES_SIZE}
     * then append the rest body request
     * Hashing steps:
     * 1. get {@link MultipartRequestOfficer#BYTES_SIZE} from the file.
     * 2. create string with both last # file's bytes.
     * 3. gzip the generated step.
     * 4. encode base64 this generated gzip.
     * 5. hash the encoded result using md5.
     *
     * @param request current processing request
     * @return hashed body
     * @throws IOException if there's any issue occurs while generating the hash.
     */
    @Nullable
    static String hashMultipartRequest(Request request) throws IOException, OutOfMemoryError {
        if (request != null && request.isMultiPartRequest()) {
            FileToUpload fileToUpload = request.getFileToUpload();
            if (fileToUpload == null) {
                return null;
            }
            File file = FileUtils.getFile(fileToUpload.getFilePath());
            String hashFile = readFileSegment(file);
            if (hashFile != null) {
                if (hashFile.isEmpty())
                    return "";
                String gzippedBody = new String(DataManipulationUtility.gzip(hashFile), Constants.UTF_8);
                // MD5 ENCODED BASE64 GZippedBody
                String encodeBase64 = DataManipulationUtility.encodeBase64(gzippedBody);

                return encodeBase64 != null ? DataManipulationUtility.md5Hash(encodeBase64) : null;
            }
        }
        return null;
    }


    /**
     * get the last {@link MultipartRequestOfficer#BYTES_SIZE}
     * using random access for better preformance
     * Listed below the covered cases:
     * 1. if file not provided then return null;
     * 2. if file less than or equals {@link MultipartRequestOfficer#BYTES_SIZE} then return the entire file
     * 3. if file size greater than {@link MultipartRequestOfficer#BYTES_SIZE} then return the last
     * {@link MultipartRequestOfficer#BYTES_SIZE}
     * <p>
     * How it's work
     * using {@link RandomAccessFile} with WRITE permission and seek till the specified size;
     *
     * @param file attached file to get required bytes for hashing them later.
     * @return the requested segment.
     * @throws IOException if any IO error occurred.
     */
    @Nullable
    @VisibleForTesting
    static String readFileSegment(@Nullable File file) throws IOException {
        if (file != null) {
            long length = file.length();
            RandomAccessFile randomAccessFile = new RandomAccessFile(file, "r");
            try {
                if (length == 0) {
                    return "";
                } else if (length <= BYTES_SIZE) {
                    byte[] byteArray = new byte[(int) length];
                    randomAccessFile.read(byteArray, 0, (int) length);
                    return new String(byteArray, Charset.forName("UTF8"));
                } else {
                    randomAccessFile.seek(file.length() - BYTES_SIZE);
                    byte[] byteArray = new byte[BYTES_SIZE];
                    randomAccessFile.read(byteArray, 0, BYTES_SIZE);
                    return new String(byteArray, Charset.forName("UTF8"));
                }
            } finally {
                randomAccessFile.close();
            }
        }
        return null;
    }

}
