package com.instabug.library.model;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Created by Hossam on 11/30/2016.
 */

public class UserAttributes implements Cacheable, Serializable {

    @VisibleForTesting
    static final String KEY_TTL = "ttl";
    @VisibleForTesting
    static final String KEY_USER_ATTRIBUTES = "user_attributes";

    private long ttl;
    @Nullable
    private HashMap<String, String> userAttributesMap;

    public long getTtl() {
        return ttl;
    }

    @Nullable
    public String get(@NonNull String key) {
        return userAttributesMap == null ? null : userAttributesMap.get(key);
    }

    public void putMap(@Nullable HashMap<String, String> userAttributesMap) {
        this.userAttributesMap = userAttributesMap;
    }

    @Nullable
    public HashMap<String, String> getMap() {
        return userAttributesMap;
    }

    public UserAttributes put(@NonNull String key, String value) {
        if (this.userAttributesMap == null) {
            userAttributesMap = new HashMap<>();
        }
        this.userAttributesMap.put(key, value);
        return this;
    }

    public void remove(@NonNull String key) {
        if (userAttributesMap != null) {
            userAttributesMap.remove(key);
        }
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject mainJsonObject = new JSONObject();
        JSONObject attributesJsonObject = new JSONObject();
        if (userAttributesMap != null) {
            for (Map.Entry<String, String> entry : userAttributesMap.entrySet()) {
                attributesJsonObject.put(entry.getKey(), entry.getValue());
            }
        }
        return mainJsonObject.put(KEY_TTL, ttl)
                .put(KEY_USER_ATTRIBUTES, attributesJsonObject)
                .toString();
    }

    @Override
    public void fromJson(String UserAttributesJsonString) throws JSONException {
        JSONObject mainJsonObject = new JSONObject(UserAttributesJsonString);
        ttl = mainJsonObject.getLong(KEY_TTL);
        //get all keys and values from UserAttributesJsonString
        JSONObject attributesJsonObject = mainJsonObject.getJSONObject(KEY_USER_ATTRIBUTES);
        Iterator<String> iterator = attributesJsonObject.keys();
        userAttributesMap = new HashMap<>();
        while (iterator.hasNext()) {
            String key = iterator.next();
            String value = attributesJsonObject.optString(key);
            userAttributesMap.put(key, value);
        }
    }

    @Override
    @NonNull
    public String toString() {
        try {
            JSONObject attributesJsonObject = new JSONObject();
            if (userAttributesMap != null) {
                for (Map.Entry<String, String> entry : userAttributesMap.entrySet()) {
                    attributesJsonObject.put(entry.getKey(), entry.getValue());
                }
            }
            return attributesJsonObject.toString();
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while parsing user attributes", e);
            return "{}";
        }
    }

    @NonNull
    public String keysAsStringJsonArray() {
        JSONArray attributesJsonArray = new JSONArray();
        if (userAttributesMap != null) {
            for (String key : userAttributesMap.keySet()) {
                attributesJsonArray.put(key);
            }
        }
        return attributesJsonArray.toString();
    }
}
