package com.instabug.library.model;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.IntDef;
import androidx.annotation.Nullable;

public class UserAttribute {

    private final int type;
    private final String key;
    private final String value;
    @Nullable
    private final String uuid;
    private final boolean anonymous;

    private UserAttribute(String key, String value, @Nullable String uuid, boolean anonymous, @Type int type) {
        this.key = key;
        this.value = value;
        this.uuid = uuid;
        this.anonymous = anonymous;
        this.type = type;
    }

    public String getKey() {
        return key;
    }

    @Nullable
    public String getUuid() {
        return uuid;
    }

    public String getValue() {
        return value;
    }

    public boolean isAnonymous() {
        return anonymous;
    }

    /**
     * @see UserAttribute.Type
     */
    @Type
    public int getType() {
        return type;
    }

    public Builder builder() {
        return new Builder(key, value)
                .uuid(uuid)
                .type(type)
                .anonymous(anonymous);
    }

    /**
     * Indicated the owner of this user attribute. Whether it got identified form the SDK or from BE
     */
    @IntDef({Type.UNSPECIFIED, Type.SDK, Type.BACKEND})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Type {
        int UNSPECIFIED = -1;
        int SDK = 0;
        int BACKEND = 1;
    }

    public static class Builder {

        private final String key;
        private final String value;
        private int type;
        @Nullable
        private String uuid;
        private boolean anonymous;

        public Builder(String key, String value) {
            this.key = key;
            this.value = value;
        }

        public Builder uuid(@Nullable String uuid) {
            this.uuid = uuid;
            return this;
        }

        /**
         * @see UserAttribute.Type
         */
        public Builder type(@Type int type) {
            this.type = type;
            return this;
        }

        public Builder anonymous(boolean anonymous) {
            this.anonymous = anonymous;
            return this;
        }

        public UserAttribute build() {
            return new UserAttribute(key, value, uuid, anonymous, type);
        }
    }
}
