package com.instabug.library.model;

import android.annotation.SuppressLint;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.internal.storage.cache.Cacheable;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public class LoggingSettings implements Cacheable {

    private static final String LEVEL_KEY = "level";
    private static final String RETENTION_DAYS_KEYS = "retention_days";
    private static final String SIZE_LIMIT_KEY = "size_limit";
    private static final String UUIDS_KEY = "uuids";
    private static final String EMAILS_KEY = "emails";
    private static final String UPLOAD_INTERVAL_KEY = "upload_interval";
    private static final String FLUSH_INTERVAL = "flush_interval";
    private static final String SINGLE_LOG_LIMIT = "single_log_limit";
    private static final String FLUSH_CHAR_LIMIT = "flush_char_limit";
    private static final String TODAY_FILE_COUNT = "today_file_count";
    private static final String KEEP_ON_DISABLED = "keep_on_sdk_disabled";
    private static final String ENCRYPTED_KEY = "encrypted";

    @VisibleForTesting
    public static final int DEFAULT_LEVEL = LogLevels.NO_LOGS;
    public static final boolean DEFAULT_ENCRYPTED = false;
    public static final int DEFAULT_RETENTION_DAY = 7;
    //Default size limit in bytes
    public static final long DEFAULT_SIZE_LIMIT = 20000;
    @VisibleForTesting
    public static final long DEFAULT_UPLOAD_INTERVAL = TimeUnit.HOURS.toSeconds(12);
    public static final int DEFAULT_FLUSH_INTERVAL_SECONDS = 2;
    public static final long DEFAULT_SINGLE_LOG_LIMIT = 4096;
    public static final long DEFAULT_FLUSH_CHAR_LIMIT = 10000;
    public static final int DEFAULT_TODAY_FILE_COUNT = 4;
    public static final boolean DEFAULT_KEEP_ON_DISABLED = false;


    private int level = DEFAULT_LEVEL;
    private boolean encrypted = DEFAULT_ENCRYPTED;
    private int retentionDay = DEFAULT_RETENTION_DAY;
    //Size limit in bytes
    private long sizeLimit = DEFAULT_SIZE_LIMIT;
    private long uploadInterval = DEFAULT_UPLOAD_INTERVAL;
    private boolean shouldClearDiskLogs = false;
    @Nullable
    private Set<String> emailSet;
    @Nullable
    private Set<String> uuidSet;
    private long flushInterval = DEFAULT_FLUSH_INTERVAL_SECONDS;
    private long singleLogLimit = DEFAULT_SINGLE_LOG_LIMIT;
    private long flushCharLimit = DEFAULT_FLUSH_CHAR_LIMIT;
    private int todayFileCount = DEFAULT_TODAY_FILE_COUNT;
    private boolean keepOnSdkDisabled = DEFAULT_KEEP_ON_DISABLED;
    private long singleFileLimit = (DEFAULT_SIZE_LIMIT / DEFAULT_TODAY_FILE_COUNT) * 1000;


    @LogLevels
    public int getLevel() {
        return level;
    }

    public void setLevel(int level) {
        this.level = level;
    }

    public int getRetentionDay() {
        return retentionDay;
    }

    public void setRetentionDay(int retentionDay) {
        this.retentionDay = retentionDay;
    }

    public long getSizeLimit() {
        return sizeLimit;
    }

    public void setSizeLimit(long sizeLimit) {
        this.sizeLimit = sizeLimit;
    }

    public long getUploadInterval() {
        return uploadInterval;
    }

    public void setUploadInterval(long uploadInterval) {
        this.uploadInterval = uploadInterval;
    }

    public void setUuidSet(@Nullable Set<String> uuidSet) {
        this.uuidSet = uuidSet;
    }

    @Nullable
    public Set<String> getUuidSet() {
        return uuidSet;
    }

    public void setEmailSet(@Nullable Set<String> emailSet) {
        this.emailSet = emailSet;
    }

    @Nullable
    public Set<String> getEmailSet() {
        return emailSet;
    }

    public long getFlushInterval() {
        return flushInterval;
    }

    public void setFlushInterval(long flushInterval) {
        this.flushInterval = flushInterval;
    }

    public long getSingleLogLimit() {
        return singleLogLimit;
    }

    public void setSingleLogLimit(long singleLogLimit) {
        this.singleLogLimit = singleLogLimit;
    }

    public long getFlushCharLimit() {
        return flushCharLimit;
    }

    public void setFlushCharLimit(long flushCharLimit) {
        this.flushCharLimit = flushCharLimit;
    }

    public int getTodayFileCount() {
        return todayFileCount;
    }

    public void setTodayFileCount(int todayFileCount) {
        this.todayFileCount = todayFileCount;
    }

    public boolean isKeepOnSdkDisabled() {
        return keepOnSdkDisabled;
    }

    public void setKeepOnSdkDisabled(boolean keepOnSdkDisabled) {
        this.keepOnSdkDisabled = keepOnSdkDisabled;
    }

    public long getSingleFileLimit() {
        return singleFileLimit;
    }

    public void setSingleFileLimit(long singleFileLimit) {
        this.singleFileLimit = singleFileLimit;
    }

    public boolean isEncrypted() {
        return encrypted;
    }

    public void setEncrypted(boolean encrypted) {
        this.encrypted = encrypted;
    }
    public boolean isShouldClearDiskLogs() {
        return shouldClearDiskLogs;
    }

    public void setShouldClearDiskLogs(boolean shouldClearDiskLogs) {
        this.shouldClearDiskLogs = shouldClearDiskLogs;
    }

    public void fromJson(@NonNull JSONObject jsonObject) throws JSONException {
        setLevel(jsonObject.optInt(LEVEL_KEY, DEFAULT_LEVEL));
        setEncrypted(jsonObject.optBoolean(ENCRYPTED_KEY, DEFAULT_ENCRYPTED));
        setRetentionDay(jsonObject.optInt(RETENTION_DAYS_KEYS, DEFAULT_RETENTION_DAY));
        setSizeLimit(jsonObject.optLong(SIZE_LIMIT_KEY, DEFAULT_SIZE_LIMIT) * 1000);
        setUploadInterval(jsonObject.optLong(UPLOAD_INTERVAL_KEY, DEFAULT_UPLOAD_INTERVAL));
        setUuidSet(fillSet(jsonObject.optJSONObject(UUIDS_KEY)));
        setEmailSet(fillSet(jsonObject.optJSONObject(EMAILS_KEY)));
        setFlushInterval(jsonObject.optInt(FLUSH_INTERVAL, DEFAULT_FLUSH_INTERVAL_SECONDS) * 1000L);
        setFlushCharLimit(jsonObject.optLong(FLUSH_CHAR_LIMIT, DEFAULT_FLUSH_CHAR_LIMIT));
        setTodayFileCount(jsonObject.optInt(TODAY_FILE_COUNT, DEFAULT_TODAY_FILE_COUNT));
        setKeepOnSdkDisabled(jsonObject.optBoolean(KEEP_ON_DISABLED, DEFAULT_KEEP_ON_DISABLED));
        setSingleLogLimit(jsonObject.optLong(SINGLE_LOG_LIMIT, DEFAULT_SINGLE_LOG_LIMIT));
        setSingleFileLimit(sizeLimit / todayFileCount);
    }

    private Set<String> fillSet(@Nullable JSONObject jsonObject) {
        HashSet<String> targetSet = new HashSet<>();
        if (jsonObject == null)
            return targetSet;
        Iterator<String> keys = jsonObject.keys();
        while (keys != null && keys.hasNext()) {
            String item = keys.next();
            targetSet.add(item);
        }
        return targetSet;
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        fromJson(new JSONObject(modelAsJson));
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject jsonObject = new JSONObject();
        jsonObject
                .put(LEVEL_KEY, getLevel())
                .put(ENCRYPTED_KEY, encrypted)
                .put(SIZE_LIMIT_KEY, getSizeLimit())
                .put(UPLOAD_INTERVAL_KEY, getUploadInterval())
                .put(RETENTION_DAYS_KEYS, getRetentionDay())
                .put(UUIDS_KEY, getUuidSet())
                .put(EMAILS_KEY, getEmailSet())
                .put(FLUSH_CHAR_LIMIT, getFlushCharLimit())
                .put(FLUSH_INTERVAL, getFlushInterval())
                .put(TODAY_FILE_COUNT, getTodayFileCount())
                .put(KEEP_ON_DISABLED, isKeepOnSdkDisabled())
                .put(SINGLE_LOG_LIMIT, getSingleLogLimit());
        return jsonObject.toString();
    }


    @IntDef({LogLevels.UNDEFINED_LEVEL, LogLevels.NO_LOGS, LogLevels.LEVEL_ONE, LogLevels.LEVEL_TWO})
    public @interface LogLevels {
        /**
         * Undefined
         */
        int UNDEFINED_LEVEL = -1;
        /**
         * No logging what so ever
         */
        int NO_LOGS = 0;
        /**
         * Errors, warning, Info
         */
        int LEVEL_ONE = 1;
        /**
         * Verbose
         */
        int LEVEL_TWO = 2;
    }


}
