package com.instabug.library.migration;

/**
 * Created by mesbah on 12/16/2015.
 */

import android.content.Context;

import com.instabug.library.Constants;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.Arrays;

import io.reactivexport.Observable;
import io.reactivexport.observers.DisposableObserver;
import io.reactivexport.schedulers.Schedulers;

public final class MigrationManager {

    private static final int TARGET_MIGRATION_VERSION = 4;
    private static final AbstractMigration[] MIGRATIONS = {new LastContactedAtMigration(),
            new V2CacheFilesMigration(),
            new LastContactedAtToLastBugAndLastChatTimeMigration(),
            new SDKBackwardMigration(),
            new SDKForwardMigration(),
            new UserAttributeMigration(),
            new VUSEncryptionMigration()};

    public static void startMigration(final Context context) {
        final Observable<AbstractMigration>[] migrations = getApplicableMigrations(context);
        if (migrations != null && migrations.length != 0) {
            Observable.merge(Arrays.asList(migrations))
                    .observeOn(Schedulers.io())
                    .subscribeOn(Schedulers.io())
                    .subscribe(new DisposableObserver<AbstractMigration>() {
                        @Override
                        public void onComplete() {
                            InstabugSDKLogger.d(Constants.LOG_TAG, "All Migrations completed, setting lastMigrationVersion to " + TARGET_MIGRATION_VERSION);
                            SettingsManager.getInstance().setLastMigrationVersion(TARGET_MIGRATION_VERSION);
                        }

                        @Override
                        public void onError(Throwable e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Migration failed" + e.getMessage());
                        }

                        @Override
                        public void onNext(AbstractMigration migration) {
                            InstabugSDKLogger.d(Constants.LOG_TAG, "Migration " + migration.getMigrationId() + " done");
                            migration.doAfterMigration();
                        }
                    });
        }
    }

    private static Observable<AbstractMigration>[] getApplicableMigrations(Context context) {
        ArrayList<Observable<AbstractMigration>> migrationObservableArrayList = new ArrayList<>();
        for (AbstractMigration migration : MIGRATIONS) {
            migration.initialize(context);
            if (shouldApplyMigration(migration)) {
                migration.doPreMigration();
                migrationObservableArrayList.add(migration.migrate());
            }
        }
        return convertArrayListToArray(migrationObservableArrayList);
    }

    private static boolean shouldApplyMigration(AbstractMigration migration) {
        final boolean shouldMigrate = migration.getMigrationVersion() <= TARGET_MIGRATION_VERSION
                && migration.shouldMigrate();
        InstabugSDKLogger.d(Constants.LOG_TAG, "Checking if should apply this migration: " + migration.getMigrationId() + ", result is " + shouldMigrate + " last migration version is " + SettingsManager.getInstance().getLastMigrationVersion() + " target migration version " + TARGET_MIGRATION_VERSION);
        return shouldMigrate;
    }

    private static Observable[] convertArrayListToArray(ArrayList<Observable<AbstractMigration>> migrationObservableArrayList) {
        Observable[] observables = new Observable[migrationObservableArrayList.size()];
        for (int i = 0; i < migrationObservableArrayList.size(); i++) {
            observables[i] = migrationObservableArrayList.get(i);
        }
        return observables;
    }
}
