package com.instabug.library.internal.storage.cache.user;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEntity.COLUMN_LAST_SEEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEntity.COLUMN_SESSION_COUNT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEntity.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.UserEntity.TABLE_NAME;

import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.dataretention.RetentionContract;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.model.User;
import com.instabug.library.util.InstabugSDKLogger;

public class UserDbHelper {
    
    /**
     * Inserting a userAttribute in the database
     */
    public static synchronized long insert(User user) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "inserting user to DB");
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            if (db.queryNumEntries(TABLE_NAME) >= RetentionContract.USER_DATA.maxNumberOfEntries()) {
                deleteLeastActiveUser(db);
            }
            ContentValues values = new ContentValues();
            values.put(COLUMN_SESSION_COUNT, user.getSessionCount());
            values.put(COLUMN_LAST_SEEN, user.getLastSeen());
            values.put(COLUMN_UUID, user.getUuid());
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflict(TABLE_NAME, null, values);
            if (rowId == -1) {
                update(user);
            }
            db.setTransactionSuccessful();
            return rowId;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Error while inserting user", Constants.LOG_TAG);
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }


    /**
     * Retrieve a User by his/her uuid
     *
     * @param uuid .
     * @return a user if found otherwise it will return null
     */
    @Nullable
    public static User retrieve(String uuid) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "retrieving user by uuid");

        // When reading data one should always just get a readable database.
        SQLiteDatabaseWrapper database = DatabaseManager.getInstance().openDatabase();

        final Cursor cursor = database.query(
                // Name of the table to read from
                TABLE_NAME,
                // String array of the columns which are supposed to be read
                null,
                // The selection argument which specifies which row is read. // ? symbols are
                // parameters.
                COLUMN_UUID + " =?",
                // The actual parameters values for the selection as a String array. // ? above
                // take the value
                // from here
                new String[]{uuid},
                // GroupBy clause. Specify a column name to group similar values // in that
                // column together.
                null,
                // Having clause. When using the GroupBy clause this allows you to // specify
                // which groups to
                // include.
                null,
                // OrderBy clause. Specify a column name here to order the results
                // according to that column. Optionally append ASC or DESC to specify // an
                // ascending or
                // descending order.
                null);

        try {
            // If moveToFirst() returns false then cursor is empty
            if (cursor == null || !cursor.moveToFirst()) {
                return null;
            }
            // To increase performance first get the index of each column in the cursor
            final int sessionCountIndex = cursor.getColumnIndex(COLUMN_SESSION_COUNT);
            final int lastSeenIndex = cursor.getColumnIndex(COLUMN_LAST_SEEN);
            // Read the values of a row in the table using the indexes acquired above
            return new User(uuid, cursor.getInt(sessionCountIndex), cursor.getLong(lastSeenIndex));
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Error while retrieving user", Constants.LOG_TAG);
            return null;
        } finally {
            // Don't forget to close the Cursor once you are done to avoid memory leaks.
            // Using a try/finally like in this example is usually the best way to handle this
            // close the database
            if (cursor != null) {
                cursor.close();
            }
            database.close();
        }
    }

    public static synchronized long update(User user) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "updating user in DB");
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_UUID + " = ?";
        String[] whereArgs = new String[]{user.getUuid()};
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_SESSION_COUNT, user.getSessionCount());
            values.put(COLUMN_LAST_SEEN, user.getLastSeen());
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.update(TABLE_NAME, values, whereClause, whereArgs);
            db.setTransactionSuccessful();
            return rowId;
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Error while updating user", Constants.LOG_TAG);
            return -1;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    public static void deleteLeastActiveUser() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "deleteLeastActiveUser ");
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        try {
            db.beginTransaction();
            deleteLeastActiveUser(db);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "Error while deleting least active user", Constants.LOG_TAG);
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    private static void deleteLeastActiveUser(SQLiteDatabaseWrapper db) {
        String salStatement = "DELETE FROM " + TABLE_NAME + " WHERE " + COLUMN_LAST_SEEN
                + " = (SELECT MIN(" + COLUMN_LAST_SEEN + ") FROM " + TABLE_NAME + ")";
        db.execSQL(salStatement);
    }
}
