package com.instabug.library.internal.storage.cache;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;

import com.instabug.library.Constants;
import com.instabug.library.model.UserAttributes;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.util.HashMap;

import androidx.annotation.Nullable;

/**
 Created by Barakat on 22/03/2019
 */
public class UserAttributesCacheManager {
    public static final String USER_ATTRIBUTES_DISK_CACHE_FILE_NAME = "/user_attributes.cache";
    public static final String USER_ATTRIBUTES_DISK_CACHE_KEY = "user_attributes_disk_cache";
    public static final String USER_ATTRIBUTES_MEMORY_CACHE_KEY = "user_attributes_memory_cache";
    public static final String USER_ATTRIBUTES_CACHE_KEY = "attrs";

    /**
     * Loads user attributes cache from disk if it's not in memory
     *
     * @return in-memory cache for user attributes
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public static InMemoryCache<String, UserAttributes> getCache() {
        CacheManager.getInstance().migrateCache(USER_ATTRIBUTES_DISK_CACHE_KEY,
                USER_ATTRIBUTES_MEMORY_CACHE_KEY, new CacheManager.KeyExtractor<String, UserAttributes>() {
                    @Override
                    public String extractKey(UserAttributes userAttributes) {
                        return USER_ATTRIBUTES_CACHE_KEY;
                    }
                });
        return (InMemoryCache<String, UserAttributes>) CacheManager.getInstance().getCache(
                USER_ATTRIBUTES_MEMORY_CACHE_KEY);
    }


    /**
     * @return All user attributes in cache.
     */
    @Nullable
    public static HashMap<String, String> getAll() {
        InMemoryCache<String, UserAttributes> cache = getCache();
        UserAttributes attributes;
        if (cache != null) {
            attributes = cache.get(USER_ATTRIBUTES_CACHE_KEY);
        } else {
            attributes = null;
        }
        return attributes == null || attributes.getMap() == null || attributes.getMap().isEmpty()
                ? null : attributes.getMap();
    }

    public static void prepareCaches(final Context context) {
        PoolProvider.postIOTaskWithCheck(new Runnable() {
            @Override
            public void run() {
                prepareUserAttributesCache(context);
            }
        });
    }

    private static void prepareUserAttributesCache(Context context) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Creating UserAttributes disk cache");
        OnDiskCache<UserAttributes> userAttributeOnDiskCache = new OnDiskCache<UserAttributes>
                (context, USER_ATTRIBUTES_DISK_CACHE_KEY,
                        USER_ATTRIBUTES_DISK_CACHE_FILE_NAME,
                        UserAttributes.class);
        CacheManager.getInstance().addCache(userAttributeOnDiskCache);
    }
}
