package com.instabug.library.internal.storage.cache;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Feature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.util.ClosableUtils;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author mSobhy
 */
public class OnDiskCache<V extends Cacheable> extends Cache<String, V> {
    public static final String TAG = "OnDiskCache";
    private final File cacheFile;
    private final File cacheDir;
    private Class<V> VClass;
    private final Charset charset;

    public OnDiskCache(Context context, String cacheId, String cacheFileName, Class<V> VClass) {
        super(cacheId);
        this.VClass = VClass;
        cacheDir = context.getCacheDir();
        cacheFile = new File(cacheDir + cacheFileName);

        AtomicBoolean isFileCreated = new AtomicBoolean(false);
        if (!cacheFile.exists()) {
            try {
                boolean isFileCreatedBool = cacheFile.createNewFile();
                isFileCreated.set(isFileCreatedBool);
            } catch (IOException e) {
                InstabugSDKLogger.e(TAG, "Failed to create", e);
            }
        } else {
            // The file already exists
            isFileCreated.set(true);
        }
        if (Charset.isSupported("UTF-8")) {
            charset = Charset.forName("UTF-8");
        } else {
            charset = Charset.defaultCharset();
        }

        if (isFileCreated.get()) {
            if (!checkCacheValidity()) {
                invalidate();
            }
        }
    }

    @Override
    public List<V> getValues() {
        final List<V> values = new ArrayList<>();
        if (!cacheFile.exists()) {
            return values;
        }
      if (Instabug.isBuilt() && Instabug.getApplicationContext() != null && FileUtils.isCacheFile(cacheFile)) {
                getReadValues(values);
          }
        return values;
    }
    @NonNull
    private void getReadValues(final List<V> values)  {
        try {
               synchronized (cacheFile) {
                       BufferedReader reader = null;
                       try {
                           reader = new BufferedReader(new InputStreamReader(new FileInputStream(cacheFile),
                                   charset));
                           String line;
                           do {
                               line = reader.readLine();
                               String valueAsJson = getValue(line);
                               if (valueAsJson != null && !valueAsJson.isEmpty()) {
                                   V value = VClass.newInstance();
                                   if (InstabugFeaturesManager.getInstance().getEncryptionState() == Feature.State.ENABLED) {
                                       valueAsJson = EncryptionManager.decrypt(valueAsJson);
                                   }
                                   if (valueAsJson != null) {
                                       value.fromJson(valueAsJson);
                                       values.add(value);
                                   }
                               }
                           } while (line != null);
                       } catch (Exception e) {
                           if (e instanceof InstantiationException)
                               InstabugSDKLogger.e(TAG, "InstantiationException happened while fetching " +
                                       "values", e);
                           else if (e instanceof IllegalAccessException)
                               InstabugSDKLogger.e(TAG, "IllegalAccessException went wrong while fetching", e);
                           else if (e instanceof IOException)
                               InstabugSDKLogger.e(TAG, "IOException went wrong while fetching values", e);
                           else if (e instanceof JSONException)
                               InstabugSDKLogger.e(TAG, "JSONException went wrong while fetching values", e);
                           else
                               InstabugSDKLogger.e(TAG, e.toString(), e);

                       } finally {
                           try {
                               if (reader != null) {
                                   reader.close();
                               }
                           } catch (IOException e) {
                               InstabugSDKLogger.e(TAG, "Failed to close file reader", e);
                           }
                       }
                   }

        } catch (Throwable throwable) {
               InstabugSDKLogger.e(TAG, "Error: " + throwable.getMessage() + "Failed to read cache values");
           }


    }





    @Override
    @Nullable
    public V get(String key) {
        if (key == null) {
            InstabugSDKLogger.d(TAG, "Key equal null");
            return null;
        }
        if (!cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file doesn't exist");
            return null;
        }

        BufferedReader reader = null;
        V value = null;
        try {
            value = VClass.newInstance();
        } catch (InstantiationException e) {
            InstabugSDKLogger.e(TAG, "InstantiationException happened while fetching value for " +
                    "key " + key, e);
        } catch (IllegalAccessException e) {
            InstabugSDKLogger.e(TAG, "IllegalAccessException went wrong while fetching value for" +
                    " key " + key, e);
        }
        synchronized (cacheFile) {
            try {
                reader = new BufferedReader(new InputStreamReader(new FileInputStream(cacheFile),
                        charset));
                String line;
                do {
                    line = reader.readLine();
                    final String tempKey = getKey(line);
                    String valueAsJson = getValue(line);
                    if (tempKey != null && tempKey.equals(key) && value != null && valueAsJson != null) {
                        if (InstabugFeaturesManager.getInstance().getEncryptionState() == Feature.State.ENABLED) {
                            valueAsJson = EncryptionManager.decrypt(valueAsJson);
                        }
                        if (valueAsJson != null) {
                            value.fromJson(valueAsJson);
                            return value;
                        }
                    }
                } while (line != null);
            } catch (IOException e) {
                InstabugSDKLogger.e(TAG, "IOException went wrong while fetching value for key "
                        + key, e);
            } catch (JSONException e) {
                InstabugSDKLogger.e(TAG, "JSONException went wrong while fetching value for key " +
                        "" + key, e);
            } catch (OutOfMemoryError e) {
                InstabugCore.reportError(e,"OOM while fetching value for key " + key);
                InstabugSDKLogger.e("OOM while fetching value for key " + key, e.toString());
            } finally {
                try {
                    if (reader != null) {
                        reader.close();
                    }
                } catch (IOException e) {
                    InstabugSDKLogger.e(TAG, "Failed to close file reader", e);
                }
            }
            InstabugSDKLogger.d(TAG, "No value found for key " + key + ", returning null");
        }
        return null;
    }

    @Override
    @Nullable
    @SuppressLint("RESOURCE_LEAK")
    public V put(String key, V value) {
        if (key == null) {
            InstabugSDKLogger.d(TAG, "Key equal null");
            return null;
        }
        if (value == null) {
            InstabugSDKLogger.d(TAG, "Value equal null");
            return null;
        }
        if (!cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file doesn't exist");
            return null;
        }
        final V oldValue = get(key);
        if (oldValue != null) {
            delete(key);
        }
        OutputStreamWriter outputStreamWriter = null;
        synchronized (cacheFile) {
            try {
                outputStreamWriter = new OutputStreamWriter(new FileOutputStream(cacheFile, true),
                        charset);
                BufferedWriter bufferWriter = new BufferedWriter(outputStreamWriter);
                bufferWriter.append(key).append(":").append(value.toJson()).append("\n");
                bufferWriter.close();
            } catch (IOException e) {
                InstabugSDKLogger.e(TAG, "Something went wrong while setting value for key " +
                        key, e);
            } catch (JSONException e) {
                e.printStackTrace();
            } catch (OutOfMemoryError e) {
                InstabugCore.reportError(e, "OOM while while setting value for key "+ key);
                InstabugSDKLogger.e("OOM while while setting value for key " + key, e.toString());
            } finally {
                if (outputStreamWriter != null) {
                    ClosableUtils.closeQuietly(outputStreamWriter);
                }
            }
        }
        return oldValue != null ? oldValue : value;
    }

    @Override
    @Nullable
    @SuppressLint("RESOURCE_LEAK")
    public V delete(String key) {
        if (!cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file doesn't exist");
            return null;
        }
        File tempCacheFile = new File(cacheDir + "/cache.tmp");
        V value = null;
        try {
            value = VClass.newInstance();
        } catch (InstantiationException e) {
            InstabugSDKLogger.e(TAG, "InstantiationException happened while deleting value for " +
                    "key " + key, e);
        } catch (IllegalAccessException e) {
            InstabugSDKLogger.e(TAG, "IllegalAccessException went wrong while deleting value for" +
                    " key " + key, e);
        }
        synchronized (cacheFile) {
            BufferedReader reader = null;
            BufferedWriter writer = null;
            try {
                reader = new BufferedReader(new InputStreamReader(
                        new FileInputStream(cacheFile), charset));
                writer = new BufferedWriter(new OutputStreamWriter(
                        new FileOutputStream(tempCacheFile, true), charset));
                String line;
                do {
                    line = reader.readLine();
                    final String foundKey = getKey(line);
                    String valueAsJson = getValue(line);
                    if (foundKey != null) {
                        if (foundKey.equals(key) && value != null && valueAsJson != null) {
                            if (InstabugFeaturesManager.getInstance().getEncryptionState() == Feature.State.ENABLED) {
                                valueAsJson = EncryptionManager.decrypt(valueAsJson);
                            }
                            if (valueAsJson != null) {
                                value.fromJson(valueAsJson);
                            }
                        } else {
                            writer.write(line + "\n");
                        }
                    }
                } while (line != null);

                if (cacheFile.delete()) {
                    if (!tempCacheFile.renameTo(cacheFile)) {
                        InstabugSDKLogger.e(TAG, "Couldn't rename temp cache file");
                    }
                }
            } catch (IOException e) {
                InstabugSDKLogger.e(TAG, "IOException went wrong while deleting value for key "
                        + key, e);
            } catch (JSONException e) {
                InstabugSDKLogger.e(TAG, "JSONException went wrong while deleting value for key " +
                        "" + key, e);
            } catch (OutOfMemoryError e) {
                InstabugCore.reportError(e, "OOM while deleting value for key " + key);
                InstabugSDKLogger.e("OOM while deleting value for key " + key, e.toString());
            } finally {
                try {
                    if (reader != null) {
                        reader.close();
                    }
                    if (writer != null) {
                        writer.flush();
                        writer.close();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return value;
    }

    @Override
    public long size() {
        if (!cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file doesn't exist");
            return -1;
        }
        synchronized (cacheFile) {
            return cacheFile.getTotalSpace();
        }
    }

    @Override
    public void invalidate() {
        if (cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file  exist");
            synchronized (cacheFile) {
                cacheFile.delete();
            }
        }
        try {
            cacheFile.createNewFile();
        } catch (IOException e) {
            InstabugSDKLogger.e(TAG, "Failed to create cache file", e);
        }
    }

    public void delete() {
        if (cacheFile != null && cacheFile.exists()) {
            InstabugSDKLogger.d(TAG, "Cache file  exist");
            synchronized (cacheFile) {
                cacheFile.delete();
            }
        }
    }

    @Nullable
    public String getValue(String line) {
        if (line != null && !line.isEmpty() && line.contains(":")) {
            return line.substring(line.indexOf(":") + 1);
        } else {
            return null;
        }
    }

    @Nullable
    public String getKey(String line) {
        if (line != null && !line.isEmpty() && line.contains(":")) {
            return line.substring(0, line.indexOf(":"));
        } else {
            return null;
        }
    }

    private boolean checkCacheValidity() {
        if (cacheFile != null) {
            if (!cacheFile.exists()) {
                InstabugSDKLogger.d(TAG, "Cache file doesn't exist");
                return false;
            }
            synchronized (cacheFile) {
                BufferedReader reader = null;
                try {
                    reader = new BufferedReader(new InputStreamReader(new FileInputStream(cacheFile),
                            charset));
                    String firstLine = reader.readLine();
                    if (firstLine != null) {
                        String firstLineValue = getValue(firstLine);
                        if (firstLineValue != null && !TextUtils.isEmpty(firstLineValue)) {
                            if (InstabugFeaturesManager.getInstance().getEncryptionState() == Feature.State.ENABLED) {
                                firstLineValue = EncryptionManager.decrypt(firstLineValue);
                            }
                            if (firstLineValue != null) {
                                new JSONObject(firstLineValue);
                            }
                        }
                    }
                } catch (IOException e) {
                    InstabugSDKLogger.e(TAG, "IOException went wrong while fetching values", e);
                } catch (JSONException e) {
                    InstabugSDKLogger.e(TAG, "JSONException went wrong while fetching values", e);
                    return false;
                } catch (OutOfMemoryError e) {
                    InstabugCore.reportError(e, "OOM while fetching values from disk cache");
                    InstabugSDKLogger.e("OOM while fetching values ", e.toString());
                } finally {
                    try {
                        if (reader != null) {
                            reader.close();
                        }
                    } catch (IOException e) {
                        InstabugSDKLogger.e(TAG, "Failed to close file reader", e);
                    }
                }
                return true;
            }
        }
        return false;
    }
}
